<?php
session_start();
require_once '../config/database.php';
require_once '../config/config.php';

if (!isset($_GET['id_surat'])) {
    die("Error: ID Surat harus disertakan");
}

$id_surat = $_GET['id_surat'];

// Query untuk data surat dan semua pegawai dalam surat tugas
$query = "SELECT DISTINCT st.*, p.id_pegawai, p.nama_pegawai, p.nip, p.jabatan, p.golongan
          FROM surat_tugas st 
          INNER JOIN detail_surat_tugas dst ON st.id_surat = dst.id_surat
          INNER JOIN pegawai p ON dst.id_pegawai = p.id_pegawai
          WHERE st.id_surat = ?
          ORDER BY 
              CASE 
                  WHEN p.jabatan LIKE '%Kepala Dinas%' THEN 1
                  WHEN p.jabatan LIKE '%Sekretaris%' THEN 2
                  WHEN p.jabatan LIKE '%Kepala Bidang%' OR p.jabatan LIKE '%Kabid%' THEN 3
                  WHEN p.jabatan LIKE '%Kepala Sub Bagian%' OR p.jabatan LIKE '%Kasubbag%' THEN 4
                  WHEN p.jabatan LIKE '%Kepala Seksi%' OR p.jabatan LIKE '%Kasi%' THEN 5
                  WHEN p.jabatan LIKE '%Pengawas%' THEN 6
                  WHEN p.jabatan LIKE '%Sopir%' THEN 7
                  ELSE 8
              END,
              CASE 
                  WHEN p.golongan LIKE 'PEMBINA UTAMA MUDA (IV/C)' THEN 1
                  WHEN p.golongan LIKE 'PEMBINA TK I (IV/B)' THEN 2
                  WHEN p.golongan LIKE 'PEMBINA (IV/A)' THEN 3
                  WHEN p.golongan LIKE 'PENATA TK I (III/D)' THEN 4
                  WHEN p.golongan LIKE 'PENATA (III/C)' THEN 5
                  WHEN p.golongan LIKE 'PENATA MUDA TK I (III/B)' THEN 6
                  WHEN p.golongan LIKE 'PENATA MUDA (III/A)' THEN 7
                  WHEN p.golongan LIKE 'PENGATUR TK I (II/D)' THEN 8
                  WHEN p.golongan LIKE 'PENGATUR (II/C)' THEN 9
                  WHEN p.golongan LIKE 'PENGATUR MUDA TK I (II/B)' THEN 10
                  WHEN p.golongan LIKE 'PENGATUR MUDA (II/A)' THEN 11
                  ELSE 12
              END,
              p.nama_pegawai ASC";

$stmt = mysqli_prepare($conn, $query);
mysqli_stmt_bind_param($stmt, "i", $id_surat);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);

$pegawai_list = [];
$data = null;

while ($row = mysqli_fetch_assoc($result)) {
    if ($data === null) {
        $data = $row;
    }
    if ($row['id_pegawai']) {
        // Debug informasi pegawai
        error_log("Pegawai ditemukan - ID: " . $row['id_pegawai'] . ", Nama: " . $row['nama_pegawai']);
        $pegawai_list[] = $row;
    }
}

if (!$data) {
    die("Data surat tidak ditemukan");
}

if (empty($pegawai_list)) {
    die("Tidak ada data pegawai yang ditemukan untuk surat ini");
}

// Debug jumlah pegawai
error_log("Total pegawai yang akan diproses: " . count($pegawai_list));

// Fungsi untuk mendapatkan rincian biaya
function getRincianBiaya($conn, $id_surat, $id_pegawai) {
    $rincian_list = [];
    $total = 0;

    // Debug untuk melihat parameter yang diterima
    error_log("ID Surat: " . $id_surat . ", ID Pegawai: " . $id_pegawai);

    // Query untuk mengambil data dari rincian_biaya_riil
    $query_riil = "SELECT * FROM rincian_biaya_riil 
                  WHERE id_surat = ? AND id_pegawai = ?";
    $stmt = mysqli_prepare($conn, $query_riil);
    mysqli_stmt_bind_param($stmt, "ii", $id_surat, $id_pegawai);
    mysqli_stmt_execute($stmt);
    $result_riil = mysqli_stmt_get_result($stmt);

    if (mysqli_num_rows($result_riil) > 0) {
        error_log("Menggunakan data dari rincian_biaya_riil");
        while ($row = mysqli_fetch_assoc($result_riil)) {
            $rincian = array(
                'uraian' => $row['uraian'] ?? '',
                'keterangan' => $row['keterangan'] ?? '',
                'jumlah' => $row['jumlah'] ?? 0
            );
            $rincian_list[] = $rincian;
            $total += (float)($rincian['jumlah']);
        }
    }

    error_log("Total rincian yang ditemukan: " . count($rincian_list));
    error_log("Total biaya: " . $total);

    return ['rincian' => $rincian_list, 'total' => $total];
}

// Fungsi untuk memformat angka ke dalam kata-kata bahasa Indonesia
function penyebut($nilai) {
    // Pastikan nilai adalah integer
    try {
        $nilai = (int)$nilai;
    } catch (Exception $e) {
        return "";
    }

    if ($nilai < 0) return "minus " . trim(penyebut(-$nilai));
    if ($nilai == 0) return "";

    $nilai = abs($nilai);
    $huruf = array("", "Satu", "Dua", "Tiga", "Empat", "Lima", "Enam", "Tujuh", "Delapan", "Sembilan", "Sepuluh", "Sebelas");
    $temp = "";

    if ($nilai < 12) {
        $temp = " " . $huruf[$nilai];
    } else if ($nilai < 20) {
        $temp = penyebut($nilai - 10) . " Belas";
    } else if ($nilai < 100) {
        $temp = penyebut((int)($nilai/10)) . " Puluh" . penyebut($nilai % 10);
    } else if ($nilai < 200) {
        $temp = " Seratus" . penyebut($nilai - 100);
    } else if ($nilai < 1000) {
        $temp = penyebut((int)($nilai/100)) . " Ratus" . penyebut($nilai % 100);
    } else if ($nilai < 2000) {
        $temp = " Seribu" . penyebut($nilai - 1000);
    } else if ($nilai < 1000000) {
        $temp = penyebut((int)($nilai/1000)) . " Ribu" . penyebut($nilai % 1000);
    } else if ($nilai < 1000000000) {
        $temp = penyebut((int)($nilai/1000000)) . " Juta" . penyebut($nilai % 1000000);
    }
    
    return $temp;
}

function terbilang($nilai) {
    // Pastikan nilai adalah numeric dan konversi ke integer
    if (!is_numeric($nilai)) {
        return "Nol Rupiah";
    }
    
    try {
        $nilai = (int)$nilai;
        if ($nilai == 0) {
            return "Nol Rupiah";
        }
        return trim(penyebut($nilai)) . " Rupiah";
    } catch (Exception $e) {
        return "Nol Rupiah";
    }
}

// Fungsi untuk memformat nama (kapital kecuali gelar)
function formatNama($nama) {
    // Daftar gelar yang umum
    $gelar_depan = array('dr.', 'dr ', 'drs.', 'drs ', 'ir.', 'ir ', 'h.', 'h ', 'hj.', 'hj ', 'prof.', 'prof ');
    $gelar_belakang = array(
        'S.Pd', 's.pd', 'S.PD', 'S.H', 's.h', 'S.h', 'S.E', 's.e', 'S.Sos', 's.sos',
        'S.T', 's.t', 'S.Kom', 's.kom', 'S.IP', 's.ip', 'M.Pd', 'm.pd', 'M.H', 'm.h',
        'M.T', 'm.t', 'M.Si', 'm.si', 'M.SE', 'm.se', 'M.Kom', 'm.kom', 'M.AP', 'm.ap',
        'M.M', 'm.m', 'PhD', 'Ph.D', 'ph.d'
    );
    
    // Pisahkan nama berdasarkan koma untuk memisahkan gelar belakang
    $parts = explode(',', $nama);
    $nama_utama = trim($parts[0]);
    $gelar_akhir = isset($parts[1]) ? trim($parts[1]) : '';
    
    // Cek dan pisahkan gelar depan
    $nama_tanpa_gelar = $nama_utama;
    $gelar_awal = '';
    foreach ($gelar_depan as $gelar) {
        if (stripos($nama_utama, $gelar) === 0) {
            $gelar_awal = substr($nama_utama, 0, strlen($gelar));
            $nama_tanpa_gelar = trim(substr($nama_utama, strlen($gelar)));
            break;
        }
    }
    
    // Ubah nama utama menjadi kapital
    $nama_kapital = strtoupper($nama_tanpa_gelar);
    
    // Gabungkan kembali dengan gelar
    $hasil = $gelar_awal . $nama_kapital;
    if (!empty($gelar_akhir)) {
        $hasil .= ', ' . $gelar_akhir;
    }
    
    return $hasil;
}

// Fungsi untuk konversi bulan ke bahasa Indonesia
function convertToIndonesianMonth($date) {
    $bulan = array(
        'January' => 'Januari',
        'February' => 'Februari',
        'March' => 'Maret',
        'April' => 'April',
        'May' => 'Mei',
        'June' => 'Juni',
        'July' => 'Juli',
        'August' => 'Agustus',
        'September' => 'September',
        'October' => 'Oktober',
        'November' => 'November',
        'December' => 'Desember'
    );
    
    return strtr($date, $bulan);
}

?>

<!DOCTYPE html>
<html>
<head>
    <title>Surat Pernyataan Daftar Pengeluaran Riil</title>
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyNCAyNCI+PHBhdGggZmlsbD0iIzRlNzNkZiIgZD0iTTIwLDhINFY2aDIwdjJIMjB6IE0xNSwxMEg0djJoMTFWMTB6IE00LDE2aDd2Mkg0VjE2eiBNMjEuNywxMy4zTDIwLjMsMTJsMS40LTEuNEMyMi4xLDEwLjIsMjIuMSw5LjgsMjEuNyw5LjRsLTEuMS0xLjFDMjAuMiw3LjksMTkuOCw3LjksMTkuNCw4LjNMMTgsMTBsLTEuNC0xLjRjLTAuNC0wLjQtMS0wLjQtMS40LDBsLTcuNSw3LjVjLTAuMiwwLjItMC4zLDAuNC0wLjMsMC43djIuOWMwLDAuNSwwLjQsMC45LDAuOSwwLjloMi45YzAuMywwLDAuNS0wLjEsMC43LTAuM2w3LjUtNy41QzIyLjEsMTQuMywyMi4xLDEzLjcsMjEuNywxMy4zeiIvPjwvc3ZnPg==">
    
    <style>
        @page {
            size: 215.9mm 330mm !important; /* Ukuran F4 */
            margin: 5mm 14mm 12.5mm 14mm; /* Mengurangi margin atas */
        }
        
        html {
            display: flex;
            justify-content: center;
            width: 215.9mm;
            min-height: 330mm;
            margin: 0 auto;
        }
        
        body {
            font-family: Times New Roman, serif;
            line-height: 1.4; /* Mengurangi line height */
            font-size: 12pt;
            margin: 0 auto;
            padding: 0;
            width: 187.9mm;
            min-height: 330mm;
            transform-origin: 0 0;
            transform: none !important;
        }
        
        @media print {
            @page {
                size: 215.9mm 330mm !important;
                margin: 5mm 14mm 12.5mm 14mm;
            }
            
            html, body {
                width: 215.9mm !important;
                height: 330mm !important;
                transform: none !important;
            }

            body {
                margin: 0 auto;
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }

            #printSection {
                position: absolute;
                left: 0;
                top: 0;
                width: 100%;
                height: 100%;
            }

            .page-container {
                page-break-after: always;
                page-break-inside: avoid;
                break-after: page;
            }

            .page-container:last-child {
                page-break-after: avoid;
                break-after: avoid;
            }
        }

        .page-container {
            page-break-after: always;
            position: relative;
        }
        .page-container:last-child {
            page-break-after: avoid;
        }
        .kop-surat {
            text-align: center;
            margin-bottom: 10px;
            position: relative;
            width: 100%;
            height: 110px;
            page-break-inside: avoid;
        }
        .logo {
            position: absolute;
            left: 30px;
            top: 0;
            width: 90px;
            height: 90px;
            z-index: 1;
        }
        .logo img {
            width: 100%;
            height: 100%;
            object-fit: contain;
        }
        .kop-text {
            text-align: center;
            padding-top: 5px;
        }
        .kop-text h2 {
            margin: 0;
            font-size: 16pt;
            font-weight: bold;
            line-height: 1.3;
        }
        .kop-text p {
            margin: 0;
            font-size: 12pt;
            line-height: 1.2;
        }
        .garis {
            border-bottom: 2px solid black;
            margin: 5px 0;
        }
        .judul {
            text-align: center;
            margin: 15px 0;
        }
        .judul h2 {
            margin: 3px 0;
            font-size: 14pt;
            font-weight: bold;
        }
        .content {
            margin: 10px 0; /* Mengurangi margin */
            line-height: 1.3;
        }
        .identitas-row {
            display: flex;
            margin: 3px 0; /* Mengurangi margin */
            line-height: 1.3;
        }
        .identitas-label {
            width: 100px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 15px 0; /* Mengurangi margin */
        }
        table, th, td {
            border: 1px solid black;
        }
        th, td {
            padding: 6px 8px; /* Mengurangi padding */
            font-size: 12pt;
            line-height: 1.3;
        }
        .text-end {
            text-align: right;
        }
        .text-center {
            text-align: center;
        }
        .footer {
            margin-top: 20px; /* Mengurangi margin */
            display: flex;
            justify-content: space-between;
        }
        .footer-section {
            width: 45%;
            text-align: center;
        }
        .footer-section p {
            margin: 0;
            line-height: 1.3;
            font-size: 12pt;
        }
    </style>
</head>
<body>
    <div id="printSection">
        <?php foreach ($pegawai_list as $index => $pegawai): 
            // Debug informasi pegawai
            echo "<!-- Debug Pegawai ke-" . ($index + 1) . " -->";
            echo "<!-- ID Pegawai: " . ($pegawai['id_pegawai'] ?? 'tidak ada') . " -->";
            echo "<!-- Nama Pegawai: " . ($pegawai['nama_pegawai'] ?? 'tidak ada') . " -->";
            
            $rincian_data = getRincianBiaya($conn, $id_surat, $pegawai['id_pegawai'] ?? 0);
            $rincian_list = $rincian_data['rincian'] ?? [];
            $total = $rincian_data['total'] ?? 0;

            // Debug informasi rincian
            echo "<!-- Jumlah rincian: " . count($rincian_list) . " -->";
            echo "<!-- Total biaya: " . $total . " -->";
        ?>
        <div class="page-container">
            <div class="kop-surat">
                <div class="logo">
                    <img src="<?php echo str_replace('rincian_biaya/', '', '../assets/img/logo.png'); ?>" alt="Logo">
                </div>
                <div class="kop-text">
                    <h2>PEMERINTAH KABUPATEN BALANGAN</h2>
                    <h2>DINAS PENDIDIKAN DAN KEBUDAYAAN</h2>
                    <p>Jl. A. Yani KM. 2,5 Kelurahan Batu Piring Telp./Fax 0526 - 2031150</p>
                    <p>Paringin Selatan - 71614</p>
                </div>
            </div>
            <div class="garis"></div>

            <div class="judul">
                <h2>SURAT PERNYATAAN</h2>
                <h2>DAFTAR PENGELUARAN RIIL</h2>
            </div>

            <div class="content">
                Yang bertandatangan dibawah ini :
                
                <div class="identitas">
                    <div class="identitas-row">
                        <div class="identitas-label">Nama</div>
                        <div>: <?= formatNama(htmlspecialchars($pegawai['nama_pegawai'] ?? '')) ?></div>
                    </div>
                    <div class="identitas-row">
                        <div class="identitas-label">NIP</div>
                        <div>: <?= htmlspecialchars($pegawai['nip'] ?? '') ?></div>
                    </div>
                    <div class="identitas-row">
                        <div class="identitas-label">Jabatan</div>
                        <div>: <?= htmlspecialchars($pegawai['jabatan'] ?? '') ?></div>
                    </div>
                </div>

                <p>Berdasarkan Surat Perintah Perjalanan Dinas ( SPPD ) tanggal <?= convertToIndonesianMonth(date('d F Y', strtotime($data['tanggal_berangkat'] ?? date('Y-m-d')))) ?> Nomor : <?= htmlspecialchars($data['no_sppd'] ?? '') ?>, dengan ini kami menyatakan dengan sesungguhnya bahwa :</p>

                <p>1. Biaya Penginapan dan transportasi pegawai dibawah ini yang tidak dapat diperoleh bukti - bukti pengeluaran berupa, meliputi :</p>
            </div>

            <table>
                <tr>
                    <th style="width: 10%">No</th>
                    <th style="width: 60%">URAIAN</th>
                    <th style="width: 30%">JUMLAH</th>
                </tr>
                <?php 
                $no = 1;
                foreach ($rincian_list as $row): 
                ?>
                    <tr>
                        <td class="text-center"><?= $no++ ?></td>
                        <td><?= htmlspecialchars($row['uraian'] ?? '') ?>
                            <?php if (!empty($row['keterangan'] ?? '')): ?>
                            <br><i>(<?= htmlspecialchars($row['keterangan']) ?>)</i>
                            <?php endif; ?>
                        </td>
                        <td class="text-end">Rp <?= number_format($row['jumlah'] ?? 0, 0, ',', '.') ?></td>
                    </tr>
                <?php endforeach; ?>
                <tr>
                    <td colspan="2" class="text-center"><strong>Total</strong></td>
                    <td class="text-end">Rp <?= number_format($total, 0, ',', '.') ?></td>
                </tr>
                <tr>
                    <td colspan="3" style="text-align: left;">
                        <strong>Terbilang : <?php 
                            $hasil_terbilang = terbilang((int)$total);
                            echo $hasil_terbilang;
                        ?></strong>
                    </td>
                </tr>
            </table>

            <div class="content">
                <p>2. Jumlah uang tersebut pada angka 1 di atas benar-benar dikeluarkan untuk pelaksanaan perjalanan dinas dimaksud dan apabila di kemudian hari terdapat kelebihan atas pembayaran, kami bersedia untuk menyetor kelebihan tersebut ke Kas Negara.</p>
                
                <p>Demikian Pernyataan ini kami buat dengan sebenarnya, untuk dipergunakan sebagaimana mestinya.</p>
            </div>

            <div class="footer">
                <div class="footer-section">
                    <p>Mengetahui / Menyetujui</p>
                    <p>Pengguna Anggaran</p>
                    <br><br><br>
                    <p><u>H. ABIJI, S.Pd, M.AP</u></p>
                    <p>NIP. 19690813 199412 1 002</p>
                </div>
                <div class="footer-section">
                    <p>Paringin, <?= convertToIndonesianMonth(date('d F Y', strtotime($data['tanggal_kembali'] ?? date('Y-m-d')))) ?></p>
                    <p>Pelaksana Perjalanan Dinas</p>
                    <br><br><br>
                    <p><u><?= formatNama(htmlspecialchars($pegawai['nama_pegawai'] ?? '')) ?></u></p>
                    <p>NIP. <?= htmlspecialchars($pegawai['nip'] ?? '') ?></p>
                </div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>

    <script>
        window.onload = function() {
            // Mengatur properti pencetakan
            const style = document.createElement('style');
            style.innerHTML = `
                @page {
                    size: 215.9mm 330mm !important;
                    margin: 5mm 14mm 12.5mm 14mm;
                }
                @media print {
                    html, body {
                        width: 215.9mm !important;
                        height: 330mm !important;
                        transform: none !important;
                    }
                }
            `;
            document.head.appendChild(style);

            setTimeout(function() {
                const printConfig = {
                    style: {
                        transform: 'none !important'
                    }
                };
                
                // Override browser print settings
                const mediaQueryList = window.matchMedia('print');
                mediaQueryList.addListener(function(mql) {
                    if (mql.matches) {
                        document.documentElement.style.setProperty('transform', 'none', 'important');
                        document.body.style.setProperty('transform', 'none', 'important');
                    }
                });

                // Set print options
                var printOptions = {
                    duplex: 'simplex',
                    pageSize: 'F4',
                    orientation: 'portrait'
                };

                window.print();
            }, 500);
        }
    </script>
</body>
</html>