<?php
session_start();
require_once '../config/database.php';
require_once '../config/config.php';

// Get user's bidang if admin_sppd
$user_bidang = '';
if ($_SESSION['level'] === 'admin_sppd') {
    $user_id = $_SESSION['user_id'];
    $stmt = mysqli_prepare($conn, "SELECT bidang FROM users WHERE id = ?");
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, "i", $user_id);
        mysqli_stmt_execute($stmt);
        $result_user = mysqli_stmt_get_result($stmt);
        if ($user_data = mysqli_fetch_assoc($result_user)) {
            $user_bidang = $user_data['bidang'];
        }
        mysqli_stmt_close($stmt);
    }
}

// Query untuk mengambil data surat tugas dengan detail pegawai
$base_query = "SELECT st.*, 
          GROUP_CONCAT(CONCAT(p.id_pegawai, ':', p.nama_pegawai) SEPARATOR '||') as pegawai_data,
          rb.id_rincian
          FROM surat_tugas st 
          LEFT JOIN detail_surat_tugas dst ON st.id_surat = dst.id_surat
          LEFT JOIN pegawai p ON dst.id_pegawai = p.id_pegawai 
          LEFT JOIN rincian_biaya rb ON st.id_surat = rb.id_surat
          LEFT JOIN users u ON st.created_by = u.id
          WHERE st.is_bimtek = 1";  // Tambahkan kondisi untuk bimtek

// Add WHERE clause for admin_sppd
if ($_SESSION['level'] === 'admin_sppd' && !empty($user_bidang)) {
    $query = $base_query . " AND u.bidang = ? GROUP BY st.id_surat ORDER BY st.tanggal_berangkat DESC";
    $stmt = mysqli_prepare($conn, $query);
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, "s", $user_bidang);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
    } else {
        die("Error preparing statement: " . mysqli_error($conn));
    }
} else {
    $query = $base_query . " GROUP BY st.id_surat ORDER BY st.tanggal_berangkat DESC";
    $result = mysqli_query($conn, $query);
}

if (!$result) {
    die("Error query: " . mysqli_error($conn));
}

// Modifikasi fungsi penyebut untuk menangani konversi float ke int dengan benar
function penyebut($nilai) {
    $nilai = floor($nilai); // Konversi ke integer dengan membulatkan ke bawah
    $nilai = abs($nilai);
    $huruf = array("", "satu", "dua", "tiga", "empat", "lima", "enam", "tujuh", "delapan", "sembilan", "sepuluh", "sebelas");
    $temp = "";
    if ($nilai < 12) {
        $temp = " ". $huruf[$nilai];
    } else if ($nilai < 20) {
        $temp = penyebut($nilai - 10). " belas";
    } else if ($nilai < 100) {
        $temp = penyebut(floor($nilai/10))." puluh". penyebut($nilai % 10);
    } else if ($nilai < 200) {
        $temp = " seratus" . penyebut($nilai - 100);
    } else if ($nilai < 1000) {
        $temp = penyebut(floor($nilai/100)) . " ratus" . penyebut($nilai % 100);
    } else if ($nilai < 2000) {
        $temp = " seribu" . penyebut($nilai - 1000);
    } else if ($nilai < 1000000) {
        $temp = penyebut(floor($nilai/1000)) . " ribu" . penyebut($nilai % 1000);
    } else if ($nilai < 1000000000) {
        $temp = penyebut(floor($nilai/1000000)) . " juta" . penyebut($nilai % 1000000);
    } else if ($nilai < 1000000000000) {
        $temp = penyebut(floor($nilai/1000000000)) . " milyar" . penyebut(fmod($nilai,1000000000));
    } else if ($nilai < 1000000000000000) {
        $temp = penyebut(floor($nilai/1000000000000)) . " trilyun" . penyebut(fmod($nilai,1000000000000));
    }     
    return $temp;
}

function terbilang($nilai) {
    if($nilai<0) {
        $hasil = "minus ". trim(penyebut($nilai));
    } else {
        $hasil = trim(penyebut($nilai));
    }     
    return $hasil . " rupiah";
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Daftar Rincian Biaya Bimtek - SPPD System</title>
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyNCAyNCI+PHBhdGggZmlsbD0iIzRlNzNkZiIgZD0iTTIwLDhINFY2aDIwdjJIMjB6IE0xNSwxMEg0djJoMTFWMTB6IE00LDE2aDd2Mkg0VjE2eiBNMjEuNywxMy4zTDIwLjMsMTJsMS40LTEuNEMyMi4xLDEwLjIsMjIuMSw5LjgsMjEuNyw5LjRsLTEuMS0xLjFDMjAuMiw3LjksMTkuOCw3LjksMTkuNCw4LjNMMTgsMTBsLTEuNC0xLjRjLTAuNC0wLjQtMS0wLjQtMS40LDBsLTcuNSw3LjVjLTAuMiwwLjItMC4zLDAuNC0wLjMsMC43djIuOWMwLDAuNSwwLjQsMC45LDAuOSwwLjhoMi45YzAuMywwLDAuNS0wLjEsMC43LTAuM2w3LjUtNy41QzIyLjEsMTQuMywyMi4xLDEzLjcsMjEuNywxMy4zeiIvPjwvc3ZnPg==">
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    
    <!-- DataTables CSS -->
    <link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/responsive/2.5.0/css/responsive.bootstrap5.min.css">
    
    <style>
        :root {
            --primary-gradient: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
            --bg-gradient: linear-gradient(135deg, rgb(245, 255, 254) 0%, rgb(153, 224, 238) 100%);
        }

        body {
            background: var(--bg-gradient) !important;
            font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
            margin: 0;
            padding: 0;
            min-height: 100vh;
            overflow-x: hidden;
        }

        .container-fluid {
            padding: 20px;
            width: 100%;
            max-width: 100%;
        }

        .header-wrapper {
            background: var(--primary-gradient);
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
            box-shadow: 0 4px 15px rgba(78,115,223,0.1);
        }

        .header-wrapper .breadcrumb-item + .breadcrumb-item::before {
            color: rgba(255, 255, 255, 0.5);
        }

        .card {
            background: white;
            backdrop-filter: blur(10px);
            border: none;
            border-radius: 15px;
            box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.15);
            overflow: hidden;
            margin-bottom: 2rem;
        }

        .card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 28px rgba(0, 0, 0, 0.12);
        }

        .card-body {
            padding: 2rem;
        }

        .table {
            margin-bottom: 0;
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }

        .table thead th {
            background: white !important;
            color: #1e293b;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.85rem;
            letter-spacing: 0.5px;
            padding: 1rem !important;
            border-bottom: 2px solid #e2e8f0;
        }

        .table tbody td {
            padding: 1rem !important;
            vertical-align: top;
            background: white;
            border-bottom: 1px solid #e2e8f0;
        }

        .table tbody tr:hover td {
            background-color: rgba(14, 165, 233, 0.04);
            transition: background-color 0.2s ease;
        }

        .table-responsive {
            border-radius: 15px;
            overflow: hidden;
        }

        .pegawai-wrapper {
            background: white;
            border: 1px solid rgba(14, 165, 233, 0.1);
            border-radius: 8px;
            padding: 0.5rem;
            box-shadow: 0 2px 8px rgba(14, 165, 233, 0.05);
        }

        .pegawai-list {
            margin: 0;
        }

        .pegawai-item {
            padding: 0.5rem 0.75rem;
            border-radius: 6px;
            margin-bottom: 0.25rem;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 0.5rem;
            width: 100%;
        }

        .pegawai-hidden {
            display: none !important;
        }

        .pegawai-wrapper.expanded .pegawai-hidden {
            display: flex !important;
            animation: fadeIn 0.3s ease;
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .lihat-semua {
            display: inline-flex;
            align-items: center;
            padding: 0.5rem;
            color: #0ea5e9;
            text-decoration: none;
            font-size: 0.875rem;
            font-weight: 500;
            cursor: pointer;
            border-radius: 4px;
            transition: all 0.2s ease;
            margin-top: 0.5rem;
            user-select: none;
        }

        .lihat-semua:hover {
            background-color: rgba(14, 165, 233, 0.1);
        }

        .lihat-semua i {
            transition: transform 0.3s ease;
            margin-right: 4px;
        }

        .pegawai-wrapper.expanded .lihat-semua i {
            transform: rotate(180deg);
        }

        .toggle-count {
            background-color: rgba(14, 165, 233, 0.1);
            padding: 2px 6px;
            border-radius: 4px;
            font-size: 0.75rem;
            margin-left: 4px;
            color: #0ea5e9;
        }

        .pegawai-name {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.875rem;
            color: #334155;
            font-weight: 500;
            flex: 1;
            min-width: 0;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }

        .pegawai-number {
            color: #64748b;
            font-size: 0.75rem;
            font-weight: 600;
            background: rgba(100, 116, 139, 0.1);
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            min-width: 24px;
            text-align: center;
        }

        .check-icon {
            color: #10b981;
            font-size: 0.875rem;
        }

        .btn-rincian {
            background: rgba(14, 165, 233, 0.1);
            color: #0ea5e9;
            border: none;
            padding: 0.35rem 0.75rem;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 0.35rem;
            transition: all 0.2s ease;
            text-decoration: none;
            white-space: nowrap;
            flex-shrink: 0;
        }

        .btn-rincian:hover {
            background: #0ea5e9;
            color: white;
            transform: translateY(-1px);
        }

        .btn-rincian i {
            font-size: 0.875rem;
        }

        .lihat-semua {
            display: flex;
            align-items: center;
            padding: 0.5rem 0.75rem;
            color: #0ea5e9;
            text-decoration: none;
            font-size: 0.75rem;
            font-weight: 500;
            transition: all 0.2s ease;
            gap: 0.35rem;
            margin-top: 0.25rem;
        }

        .lihat-semua:hover {
            color: #0284c7;
        }

        .lihat-semua i {
            font-size: 0.875rem;
            transition: transform 0.2s ease;
        }

        .pegawai-wrapper.expanded .lihat-semua i {
            transform: rotate(180deg);
        }

        .toggle-count {
            color: #64748b;
            background: rgba(100, 116, 139, 0.1);
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-size: 0.75rem;
        }

        .pegawai-hidden {
            display: none !important;
        }

        .pegawai-wrapper.expanded .pegawai-hidden {
            display: flex !important;
        }

        .pegawai-wrapper.expanded .toggle-count {
            display: none;
        }

        .pegawai-wrapper .lihat-semua {
            cursor: pointer;
            color: #0ea5e9;
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 0;
            font-size: 0.875rem;
        }

        .pegawai-wrapper .lihat-semua:hover {
            color: #0284c7;
        }

        .pegawai-wrapper .lihat-semua i {
            transition: transform 0.3s ease;
        }

        .pegawai-wrapper.expanded .lihat-semua i {
            transform: rotate(180deg);
        }

        .btn-toggle-pegawai {
            display: inline-flex;
            align-items: center;
            gap: 4px;
            background: none;
            border: none;
            color: #0ea5e9;
            padding: 4px 8px;
            font-size: 0.85rem;
            cursor: pointer;
            margin-top: 4px;
            border-radius: 4px;
            transition: all 0.3s ease;
        }

        .btn-toggle-pegawai:hover {
            background-color: rgba(14, 165, 233, 0.1);
        }

        .btn-toggle-pegawai i {
            transition: transform 0.3s ease;
        }

        .btn-toggle-pegawai.active i {
            transform: rotate(180deg);
        }

        .pegawai-wrapper.expanded .btn-toggle-pegawai i {
            transform: rotate(180deg);
        }

        .pegawai-wrapper.expanded .toggle-count {
            display: none;
        }

        .pegawai-wrapper:not(.expanded) .toggle-text:before {
            content: '';  /* Hapus content yang menambahkan teks "Lihat Semua" */
        }

        .toggle-count {
            color: #6c757d;
            font-size: 0.8rem;
        }

        .toggle-text {
            color: #0ea5e9;
        }

        .lihat-semua {
            display: inline-flex;
            align-items: center;
            gap: 4px;
            color: #0ea5e9;
            text-decoration: none;
            font-size: 0.85rem;
            margin-top: 4px;
        }

        .lihat-semua:hover {
            text-decoration: underline;
        }

        .lihat-semua i {
            font-size: 0.9rem;
        }

        .rincian-wrapper {
            position: relative;
            padding-right: 30px;
        }

        .rincian-list {
            max-height: 100%;
            overflow: hidden;
            transition: max-height 0.3s ease;
        }

        .rincian-item {
            padding: 4px 0;
            color: #2c3e50;
            font-weight: 500;
        }

        .rincian-hidden {
            display: none;
        }

        .rincian-expanded .rincian-hidden {
            display: block;
        }

        .pegawai-actions {
            display: flex;
            gap: 0.5rem;
        }

        .btn {
            padding: 0.5rem 1rem;
            border-radius: 0.5rem;
            font-weight: 500;
            transition: all 0.2s;
        }

        .btn-primary,
        .btn-soft-info,
        .btn-soft-warning,
        .btn-soft-danger,
        .btn-soft-primary {
            background: var(--primary-gradient);
            color: white;
            border: none;
            transition: all 0.3s ease;
        }

        .btn-primary:hover,
        .btn-soft-info:hover,
        .btn-soft-warning:hover,
        .btn-soft-danger:hover,
        .btn-soft-primary:hover {
            background: linear-gradient(135deg, #0284c7 0%, #14b8a6 100%);
            color: white;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(14, 165, 233, 0.2);
        }

        .input-group {
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            border-radius: 0.5rem;
        }

        #tableSearch {
            border-radius: 0 0.5rem 0.5rem 0;
            border: 1px solid rgba(0, 0, 0, 0.1);
            border-left: none;
            padding: 0.75rem 1rem;
            font-size: 1rem;
        }

        #tableSearch:focus {
            box-shadow: none;
            border-color: #4e73df;
        }

        .input-group-text {
            border-radius: 0.5rem 0 0 0.5rem;
            border: 1px solid rgba(0, 0, 0, 0.1);
            background: var(--primary-gradient);
            color: white;
            padding: 0 1rem;
        }

        .dataTables_wrapper .row {
            margin: 1rem 0;
            align-items: center;
        }

        .dataTables_length select {
            border-radius: 0.5rem;
            padding: 0.375rem 2rem 0.375rem 0.75rem;
            border: 1px solid rgba(0, 0, 0, 0.1);
            background-color: white;
        }

        .dataTables_filter {
            display: none;
        }

        .dataTables_info {
            color: #6c757d;
        }

        .page-item .page-link {
            border-radius: 0.5rem;
            margin: 0 0.2rem;
            color: #4e73df;
        }

        .page-item.active .page-link {
            background: var(--primary-gradient);
            border: none;
        }

        .dropdown-menu {
            border: none;
            border-radius: 0.5rem;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.1);
            padding: 0.5rem;
        }

        .dropdown-item {
            border-radius: 0.375rem;
            padding: 0.5rem 1rem;
            transition: all 0.2s;
        }

        .dropdown-item:hover {
            background-color: rgba(78, 115, 223, 0.1);
            color: #4e73df;
        }

        .badge {
            display: inline-block;
            padding: 0.35rem 0.65rem;
            font-size: 0.875rem;
            font-weight: 500;
            line-height: 1;
            text-align: left;
            white-space: nowrap;
            vertical-align: baseline;
            border-radius: 4px;
            margin-bottom: 0.25rem;
            width: 100%;
        }

        .badge.bg-info {
            background: rgba(14, 165, 233, 0.1) !important;
            color: #0ea5e9 !important;
            border: 1px solid rgba(14, 165, 233, 0.2);
        }

        .badge.bg-success {
            background: rgba(34, 197, 94, 0.1) !important;
            color: #22c55e !important;
            border: 1px solid rgba(34, 197, 94, 0.2);
        }

        .table td.text-center {
            padding: 0.50rem;
        }

        .input-group-sm {
            max-width: 300px;
        }

        .input-group-sm .input-group-text {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
            border-radius: 0.375rem 0 0 0.375rem;
            border: 1px solid rgba(0, 0, 0, 0.1);
            background: var(--primary-gradient);
            color: white;
        }

        .input-group-sm #tableSearch {
            height: 31px;
            font-size: 0.875rem;
            border-radius: 0 0.375rem 0.375rem 0;
            border: 1px solid rgba(0, 0, 0, 0.1);
            border-left: none;
            padding: 0.25rem 0.5rem;
        }

        .input-group-sm #tableSearch:focus {
            box-shadow: none;
            border-color: #4e73df;
        }

        .status-dot {
            width: 8px;
            height: 8px;
            border-radius: 50%;
            margin-right: 8px;
            position: relative;
            display: inline-block;
        }

        .status-dot::after {
            content: '';
            position: absolute;
            width: 100%;
            height: 100%;
            border-radius: 50%;
            animation: pulse 2s infinite;
            opacity: 0.5;
        }

        .status-done {
            background-color: #198754;
        }

        .status-done::after {
            background-color: #198754;
        }

        .status-pending {
            background-color: #ffc107;
        }

        .status-pending::after {
            background-color: #ffc107;
        }

        @keyframes pulse {
            0% {
                transform: scale(1);
                opacity: 0.5;
            }
            70% {
                transform: scale(1.5);
                opacity: 0;
            }
            100% {
                transform: scale(1);
                opacity: 0;
            }
        }

        .status-dot:hover::before {
            content: attr(title);
            position: absolute;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            white-space: nowrap;
            top: -30px;
            left: 50%;
            transform: translateX(-50%);
            z-index: 1000;
        }

        .btn-eye {
            width: 32px;
            height: 32px;
            padding: 0;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            border-radius: 6px;
            background: rgba(14, 165, 233, 0.1);
            color: #0ea5e9;
            border: 1px solid rgba(14, 165, 233, 0.2);
            transition: all 0.2s ease;
        }

        .btn-eye:hover {
            background: #0ea5e9;
            color: white;
        }

        .action-container {
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 1rem;
            display: flex;
            gap: 0.75rem;
            align-items: center;
        }

        .action-container .btn {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .dropdown-menu {
            padding: 8px;
            border: none;
            border-radius: 8px;
            box-shadow: 0 4px 16px rgba(14, 165, 233, 0.1);
            min-width: 200px;
        }

        .dropdown-item {
            padding: 8px 12px;
            border-radius: 6px;
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 0.875rem;
            color: #0ea5e9;
            transition: all 0.2s ease;
        }

        .dropdown-item:hover {
            background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
            color: white;
        }

        .dropdown-item i {
            font-size: 1rem;
        }

        .action-container .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(14, 165, 233, 0.15);
        }

        .action-container .btn:active {
            transform: translateY(1px);
        }

        .action-container > * {
            margin: 0 !important;
        }

        .dropdown-menu.show {
            animation: fadeIn 0.2s ease;
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .btn-soft-info.btn-xs {
            background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
            color: white;
            border: none;
            padding: 4px 12px;
            height: 24px;
            font-size: 0.75rem;
            display: inline-flex;
            align-items: center;
            border-radius: 4px;
        }

        .btn-soft-info.btn-xs:hover {
            transform: translateY(-1px);
            box-shadow: 0 2px 6px rgba(14, 165, 233, 0.15);
        }

        .pegawai-container {
            position: relative;
            padding-right: 30px;
        }

        .pegawai-hidden {
            display: none;
        }

        .btn-toggle-pegawai {
            background: none;
            border: none;
            color: #0ea5e9;
            padding: 4px;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 5px;
            font-size: 0.85rem;
            margin-top: 5px;
        }

        .btn-toggle-pegawai:hover {
            color: #0891ce;
        }

        .btn-toggle-pegawai .toggle-count {
            color: #6c757d;
            font-size: 0.8rem;
        }

        .pegawai-container.expanded .pegawai-hidden {
            display: flex;
        }

        .pegawai-container.expanded .toggle-count {
            display: none;
        }

        .pegawai-container.expanded .toggle-text {
            display: none;
        }

        .d-flex.flex-column.align-items-start {
            width: 100%;
            min-width: 200px;
        }

        .badge {
            display: block;
            width: 100%;
            text-align: left;
            padding: 0.5rem 0.75rem;
            margin-bottom: 0.25rem;
            font-size: 0.875rem;
            white-space: nowrap;
            border-radius: 4px;
        }

        .badge.bg-info {
            background: rgba(14, 165, 233, 0.15) !important;
            color: #0ea5e9 !important;
            font-weight: 500;
        }

        .badge.bg-success {
            background: rgba(16, 185, 129, 0.15) !important;
            color: #10b981 !important;
            font-weight: 600;
        }

        .table td:last-child {
            min-width: 220px;
            width: 220px;
        }

        .badge span {
            display: inline-block;
            min-width: 120px;
        }

        .text-wrap {
            white-space: normal;
            word-wrap: break-word;
            max-width: 100%;
            line-height: 1.5;
        }

        @media (max-width: 768px) {
            .container-fluid {
                margin-left: 0;
                width: 100%;
                padding: 1rem;
            }
            
            body.sidebar-collapsed .container-fluid {
                margin-left: 0;
                width: 100%;
            }

            .header-wrapper {
                padding: 1.5rem;
            }

            .table td, .table th {
                padding: 0.75rem !important;
            }
            
            .badge {
                padding: 0.4em 0.8em;
                font-size: 0.8rem;
            }
            
            .btn-eye {
                width: 28px;
                height: 28px;
            }
        }

        .table {
            margin-bottom: 0;
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }

        .table td {
            vertical-align: top;
            padding: 1rem;
        }

        .table td > div {
            word-wrap: break-word;
            overflow-wrap: break-word;
        }

        @media (max-width: 1200px) {
            .table th[style*="width: 35%"],
            .table td[style*="width: 35%"] {
                width: 30% !important;
            }
        }

        @media (max-width: 992px) {
            .table th[style*="width: 35%"],
            .table td[style*="width: 35%"] {
                width: 25% !important;
            }
        }

        .total-row {
            background-color: #f8f9fa;
            border-top: 2px solid #dee2e6;
        }

        .total-row td {
            padding: 1rem !important;
            font-size: 1.1rem;
        }

        .total-amount {
            color: #0ea5e9;
            font-weight: 600;
        }

        .text-end {
            text-align: right !important;
        }

        .pe-3 {
            padding-right: 1rem !important;
        }

        .no-input-row td {
            background-color: rgba(220, 53, 69, 0.1) !important;
            color: #dc3545 !important;
        }

        .no-input-row:hover td {
            background-color: rgba(220, 53, 69, 0.15) !important;
        }

        .no-input-row .badge {
            background-color: rgba(220, 53, 69, 0.15) !important;
            color: #dc3545 !important;
            border: 1px solid #dc3545;
        }

        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.05);
            transition: transform 0.2s ease, box-shadow 0.2s ease;
            margin-bottom: 2rem;
            background: white;
        }

        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }

        .table {
            margin-bottom: 0;
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }

        .table th {
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.85rem;
            letter-spacing: 0.5px;
            padding: 1.2rem 1rem !important;
            background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
            color: white;
            border: none;
        }

        .table th:first-child {
            border-top-left-radius: 12px;
        }

        .table th:last-child {
            border-top-right-radius: 12px;
        }

        .table td {
            padding: 1.2rem 1rem !important;
            vertical-align: middle;
            border-bottom: 1px solid rgba(0,0,0,0.05);
            font-size: 0.95rem;
        }

        .badge {
            padding: 0.6em 1em;
            border-radius: 8px;
            font-weight: 500;
            font-size: 0.85rem;
            letter-spacing: 0.3px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            transition: all 0.2s ease;
        }

        .badge.bg-warning {
            background: rgba(245, 158, 11, 0.15) !important;
            color: #f59e0b !important;
            border: 1px solid rgba(245, 158, 11, 0.3);
        }

        .btn {
            padding: 0.6rem 1.2rem;
            border-radius: 8px;
            font-weight: 500;
            letter-spacing: 0.3px;
            transition: all 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn i {
            font-size: 1.1rem;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }

        .header-wrapper {
            background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
            padding: 2.5rem;
            border-radius: 20px;
            margin-bottom: 2rem;
            box-shadow: 0 8px 25px rgba(14, 165, 233, 0.2);
        }

        .header-wrapper h1 {
            font-size: 1.8rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            letter-spacing: 0.5px;
        }

        .header-wrapper p {
            font-size: 1rem;
            opacity: 0.8;
            margin-bottom: 0;
        }

        .action-container {
            padding: 1rem;
            background: rgba(255, 255, 255, 0.98);
            border-radius: 12px;
            display: flex;
            gap: 0.8rem;
            align-items: center;
            margin: 0;
            border: 1px solid rgba(14, 165, 233, 0.15);
            box-shadow: 0 4px 15px rgba(14, 165, 233, 0.08);
        }

        .pegawai-wrapper {
            background: white;
            border: 1px solid rgba(14, 165, 233, 0.1);
            border-radius: 8px;
            padding: 0.5rem;
            box-shadow: 0 2px 8px rgba(14, 165, 233, 0.05);
        }

        .pegawai-list {
            margin: 0;
        }

        .pegawai-item {
            padding: 0.5rem 0.75rem;
            border-radius: 6px;
            margin-bottom: 0.25rem;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 0.5rem;
            width: 100%;
        }

        .pegawai-hidden {
            display: none !important;
        }

        .pegawai-wrapper.expanded .pegawai-hidden {
            display: flex !important;
        }

        .lihat-semua {
            display: flex;
            align-items: center;
            padding: 0.5rem 0.75rem;
            color: #0ea5e9;
            text-decoration: none;
            font-size: 0.75rem;
            font-weight: 500;
            transition: all 0.2s ease;
            gap: 0.35rem;
            margin-top: 0.25rem;
            cursor: pointer;
            user-select: none;
        }

        .lihat-semua:hover {
            color: #0284c7;
            background: rgba(14, 165, 233, 0.05);
            border-radius: 6px;
        }

        .lihat-semua i {
            font-size: 0.875rem;
            transition: transform 0.2s ease;
        }

        .toggle-count {
            color: #64748b;
            background: rgba(100, 116, 139, 0.1);
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-size: 0.75rem;
        }

        /* Maksud Styling */
        .maksud-wrapper {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .jenis-surat-badge {
            display: inline-flex;
            align-items: center;
            padding: 0.25rem 0.75rem;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 500;
            gap: 0.5rem;
            width: fit-content;
        }

        .jenis-surat-badge i {
            font-size: 0.875rem;
        }

        /* Badge colors for different types */
        .dalam-daerah-dalam-provinsi {
            background-color: rgba(34, 197, 94, 0.1);
            color: #16a34a;
            border: 1px solid rgba(34, 197, 94, 0.2);
        }

        .luar-daerah-dalam-provinsi {
            background-color: rgba(249, 115, 22, 0.1);
            color: #ea580c;
            border: 1px solid rgba(249, 115, 22, 0.2);
        }

        .luar-daerah-luar-provinsi {
            background-color: rgba(99, 102, 241, 0.1);
            color: #4f46e5;
            border: 1px solid rgba(99, 102, 241, 0.2);
        }

        .maksud-text {
            color: #1e293b;
            line-height: 1.5;
            font-size: 0.95rem;
        }

        /* Update table header styles */
        .table thead th {
            background: white !important;
            color: #1e293b;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.85rem;
            letter-spacing: 0.5px;
            padding: 1rem !important;
            border-bottom: 2px solid #e2e8f0;
            white-space: nowrap;
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .jenis-surat-badge {
                font-size: 0.7rem;
                padding: 0.2rem 0.5rem;
            }
            
            .maksud-text {
                font-size: 0.9rem;
            }
        }

        /* New responsive styles */
        .header-wrapper {
            padding: 2rem;
            border-radius: 15px;
            margin-bottom: 2rem;
            box-shadow: 0 4px 15px rgba(78,115,223,0.1);
        }

        .header-wrapper h1 {
            font-size: clamp(1.5rem, 4vw, 1.8rem);
            margin-bottom: 0.5rem;
        }

        .header-wrapper p {
            font-size: clamp(0.875rem, 2vw, 1rem);
        }

        .table-responsive {
            margin: 0;
            padding: 0;
            width: 100%;
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }

        .table {
            width: 100%;
            margin-bottom: 0;
        }

        .table th {
            white-space: nowrap;
            padding: 1rem !important;
        }

        .table td {
            vertical-align: middle;
            min-width: 100px;
        }

        @media (max-width: 1200px) {
            .table th[style*="width: 50%"],
            .table td[style*="width: 50%"] {
                width: 40% !important;
            }
            
            .table th[style*="width: 20%"],
            .table td[style*="width: 20%"] {
                width: 25% !important;
            }
        }

        @media (max-width: 992px) {
            .container-fluid {
                padding: 15px;
            }

            .header-wrapper {
                padding: 1.5rem;
            }

            .card {
                margin-bottom: 1rem;
            }

            .table th[style*="width: 50%"],
            .table td[style*="width: 50%"] {
                width: 35% !important;
            }
        }

        @media (max-width: 768px) {
            .container-fluid {
                padding: 10px;
            }

            .header-wrapper {
                padding: 1rem;
                margin-bottom: 1rem;
            }

            .breadcrumb {
                display: none;
            }

            .table td {
                padding: 0.75rem !important;
            }

            .pegawai-wrapper {
                padding: 0.5rem;
            }

            .pegawai-item {
                padding: 0.5rem;
                flex-direction: column;
                align-items: flex-start;
            }

            .pegawai-name {
                margin-bottom: 0.5rem;
                font-size: 0.8rem;
            }

            .btn-rincian {
                width: 100%;
                justify-content: center;
            }

            .badge {
                font-size: 0.75rem;
                padding: 0.35rem 0.5rem;
                width: 100%;
            }

            .action-container {
                flex-wrap: wrap;
                gap: 0.5rem;
            }

            .action-container .btn {
                width: 100%;
                justify-content: center;
            }

            .btn-group {
                width: 100%;
                display: flex;
                gap: 0.25rem;
            }

            .btn-group .btn {
                flex: 1;
            }
        }

        @media (max-width: 576px) {
            .container-fluid {
                padding: 8px;
            }

            .header-wrapper {
                padding: 1rem;
            }

            .table th, 
            .table td {
                padding: 0.5rem !important;
                font-size: 0.85rem;
            }

            .maksud-text {
                font-size: 0.85rem;
            }

            .jenis-surat-badge {
                font-size: 0.7rem;
                padding: 0.25rem 0.5rem;
            }

        .total-row td {
                font-size: 0.9rem;
            }

            .dropdown-menu {
                width: 100%;
                min-width: unset;
            }
        }

        /* Fix table header on mobile */
        @media (max-width: 480px) {
            .table thead th {
                position: sticky;
                top: 0;
                background: white !important;
                z-index: 1;
            }

            .table td:before {
                content: attr(data-label);
                float: left;
                font-weight: bold;
                text-transform: uppercase;
            }

            .table td:last-child {
                border-bottom: 0;
            }
        }

        /* Improve DataTables responsive behavior */
        .dataTables_wrapper .row {
            margin: 0;
            padding: 1rem 0;
        }

        .dataTables_length,
        .dataTables_filter {
            margin-bottom: 1rem;
        }

        @media (max-width: 768px) {
            .dataTables_length,
            .dataTables_filter {
                text-align: left;
                width: 100%;
            }

            .dataTables_filter input {
                width: 100%;
                margin-left: 0;
            }
        }

        /* Improve button and badge responsiveness */
        .btn {
            white-space: nowrap;
            padding: 0.5rem 1rem;
        }

        @media (max-width: 768px) {
            .btn {
                padding: 0.4rem 0.8rem;
                font-size: 0.875rem;
            }

            .btn i {
                font-size: 1rem;
            }
        }

        /* Make badges more compact on mobile */
        @media (max-width: 768px) {
            .badge {
                display: inline-block;
                width: auto;
                margin-right: 0.25rem;
                margin-bottom: 0.25rem;
            }

            .d-flex.flex-column.align-items-start {
                flex-wrap: wrap;
                flex-direction: row !important;
            }
        }

        /* Improve table scrolling on mobile */
        .table-responsive {
            scrollbar-width: thin;
            scrollbar-color: rgba(14, 165, 233, 0.2) transparent;
        }

        .table-responsive::-webkit-scrollbar {
            height: 6px;
        }

        .table-responsive::-webkit-scrollbar-track {
            background: transparent;
        }

        .table-responsive::-webkit-scrollbar-thumb {
            background-color: rgba(14, 165, 233, 0.2);
            border-radius: 20px;
        }

        /* Header and Navigation Styles */
        .header-wrapper {
            background: var(--primary-gradient);
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
            box-shadow: 0 4px 15px rgba(78,115,223,0.1);
        }

        .nav-breadcrumb {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 0.75rem;
        }

        .nav-breadcrumb a {
            color: rgba(255, 255, 255, 0.8);
            text-decoration: none;
            font-size: 0.85rem;
            display: flex;
            align-items: center;
            gap: 0.35rem;
            transition: all 0.2s ease;
        }

        .nav-breadcrumb a:hover {
            color: white;
        }

        .nav-breadcrumb i {
            font-size: 1rem;
        }

        .nav-breadcrumb .separator {
            color: rgba(255, 255, 255, 0.4);
            margin: 0 0.35rem;
        }

        .page-title {
            color: white;
            margin: 0;
            font-size: 1.4rem;
            font-weight: 600;
            line-height: 1.2;
        }

        .page-subtitle {
            color: rgba(255, 255, 255, 0.7);
            margin: 0.35rem 0 1rem 0;
            font-size: 0.9rem;
        }

        .action-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin: 1rem 0 2rem 0;
            gap: 1rem;
        }

        .back-button {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            background: white;
            color: #0ea5e9;
            border: 1px solid rgba(14, 165, 233, 0.2);
            border-radius: 8px;
            font-weight: 500;
            text-decoration: none;
            transition: all 0.2s ease;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
        }

        .back-button:hover {
            background: #0ea5e9;
            color: white;
            transform: translateY(-1px);
        }

        .back-button i {
            font-size: 1.1rem;
        }

        @media (max-width: 768px) {
            .header-wrapper {
                padding: 1.25rem;
                margin-bottom: 1rem;
            }

            .page-title {
                font-size: 1.2rem;
            }

            .page-subtitle {
                font-size: 0.85rem;
                margin-bottom: 0.75rem;
            }

            .action-bar {
                flex-direction: column;
                gap: 0.75rem;
                margin: 1rem 0;
            }

            .action-bar > * {
                width: 100%;
            }

            .back-button {
                justify-content: center;
            }
        }

        /* Navigation Menu Styles */
        .nav-menu {
            display: flex;
            flex-wrap: wrap;
            gap: 0.35rem;
            margin-top: 1rem;
            padding-top: 1rem;
            border-top: 1px solid rgba(255, 255, 255, 0.1);
        }

        .nav-menu .nav-link {
            display: flex;
            align-items: center;
            gap: 0.35rem;
            padding: 0.4rem 0.75rem;
            background: rgba(255, 255, 255, 0.1);
            color: white;
            text-decoration: none;
            border-radius: 6px;
            transition: all 0.2s ease;
            font-size: 0.85rem;
        }

        .nav-menu .nav-link:hover {
            background: rgba(255, 255, 255, 0.2);
            transform: translateY(-1px);
        }

        .nav-menu .nav-link.active {
            background: white;
            color: #0ea5e9;
            font-weight: 500;
        }

        .nav-menu .nav-link i {
            font-size: 1rem;
        }

        /* Responsive Adjustments */
        @media (max-width: 768px) {
            .header-wrapper {
                padding: 1.25rem;
                margin-bottom: 1rem;
            }

            .page-title {
                font-size: 1.2rem;
            }

            .page-subtitle {
                font-size: 0.85rem;
                margin-bottom: 0.75rem;
            }

            .nav-menu {
                gap: 0.25rem;
                margin-top: 0.75rem;
                padding-top: 0.75rem;
            }

            .nav-menu .nav-link {
                padding: 0.35rem 0.6rem;
                font-size: 0.8rem;
            }

            .nav-menu .nav-link i {
                font-size: 0.9rem;
            }
        }

        @media (max-width: 576px) {
            .header-wrapper {
                padding: 1rem;
                margin-bottom: 1rem;
            }

            .nav-menu {
                flex-wrap: nowrap;
                overflow-x: auto;
                padding-bottom: 0.5rem;
                -webkit-overflow-scrolling: touch;
                scrollbar-width: none;
            }

            .nav-menu::-webkit-scrollbar {
                display: none;
            }

            .nav-menu .nav-link {
                white-space: nowrap;
            }
        }

        /* DataTables Length Menu Styling */
        .dataTables_length {
            margin: 1rem 0;
            font-size: 0.9rem;
        }

        .dataTables_length label {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 0;
        }

        .dataTables_length select {
            padding: 0.375rem 2.25rem 0.375rem 0.75rem;
            font-size: 0.9rem;
            font-weight: 400;
            line-height: 1.5;
            color: #1e293b;
            background-color: #fff;
            background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16'%3e%3cpath fill='none' stroke='%23343a40' stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='m2 5 6 6 6-6'/%3e%3c/svg%3e");
            background-repeat: no-repeat;
            background-position: right 0.75rem center;
            background-size: 16px 12px;
            border: 1px solid rgba(14, 165, 233, 0.2);
            border-radius: 0.375rem;
            appearance: none;
            cursor: pointer;
            min-width: 80px;
        }

        .dataTables_length select:focus {
            border-color: #0ea5e9;
            outline: 0;
            box-shadow: 0 0 0 0.2rem rgba(14, 165, 233, 0.25);
        }

        /* Search Box Styling */
        .dataTables_filter {
            margin: 1rem 0;
            font-size: 0.9rem;
        }

        .dataTables_filter label {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 0;
        }

        .dataTables_filter input {
            padding: 0.375rem 0.75rem;
            font-size: 0.9rem;
            line-height: 1.5;
            color: #1e293b;
            background-color: #fff;
            border: 1px solid rgba(14, 165, 233, 0.2);
            border-radius: 0.375rem;
            min-width: 200px;
        }

        .dataTables_filter input:focus {
            border-color: #0ea5e9;
            outline: 0;
            box-shadow: 0 0 0 0.2rem rgba(14, 165, 233, 0.25);
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .dataTables_length,
            .dataTables_filter {
                width: 100%;
                margin: 0.5rem 0;
            }

            .dataTables_filter input {
                width: 100%;
                min-width: unset;
            }

            .dataTables_length select {
                min-width: 100px;
            }
        }

        /* Table Styling */
        .table {
            margin-bottom: 0;
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }

        .table thead th {
            background: #f8fafc !important;
            color: #475569;
            font-weight: 600;
            font-size: 0.875rem;
            letter-spacing: 0.025em;
            text-transform: none;
            padding: 1rem 1.25rem !important;
            border-bottom: 1px solid #e2e8f0;
            white-space: nowrap;
        }

        .table tbody tr {
            transition: all 0.2s ease;
        }

        .table tbody tr:hover {
            background-color: #f8fafc;
        }

        .table td {
            padding: 1.25rem !important;
            vertical-align: top;
            border-bottom: 1px solid #e2e8f0;
            color: #1e293b;
            font-size: 0.9375rem;
        }

        /* Card Styling */
        .card {
            border: none;
            border-radius: 16px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            background: white;
            margin-bottom: 2rem;
        }

        .card-body {
            padding: 1.5rem;
        }

        /* Badge Styling */
        .badge {
            padding: 0.5rem 0.75rem;
            border-radius: 6px;
            font-weight: 500;
            font-size: 0.875rem;
            letter-spacing: 0.01em;
            border: 1px solid transparent;
        }

        .badge.bg-info {
            background-color: rgba(14, 165, 233, 0.1) !important;
            color: #0284c7 !important;
            border-color: rgba(14, 165, 233, 0.2);
        }

        .badge.bg-success {
            background-color: rgba(34, 197, 94, 0.1) !important;
            color: #16a34a !important;
            border-color: rgba(34, 197, 94, 0.2);
        }

        /* Maksud Section Styling */
        .maksud-wrapper {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
        }

        .jenis-surat-badge {
            display: inline-flex;
            align-items: center;
            padding: 0.375rem 0.75rem;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 500;
            gap: 0.375rem;
            width: fit-content;
        }

        .maksud-text {
            color: #334155;
            line-height: 1.5;
            font-size: 0.9375rem;
        }

        /* Pegawai List Styling */
        .pegawai-wrapper {
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 0.75rem;
        }

        .pegawai-item {
            padding: 0.625rem;
            border-radius: 6px;
            background: white;
            margin-bottom: 0.5rem;
            border: 1px solid #e2e8f0;
            transition: all 0.2s ease;
        }

        .pegawai-item:hover {
            border-color: #cbd5e1;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
        }

        .pegawai-name {
            color: #334155;
            font-size: 0.875rem;
            font-weight: 500;
        }

        .btn-rincian {
            padding: 0.375rem 0.75rem;
            font-size: 0.75rem;
            border-radius: 6px;
            background: rgba(14, 165, 233, 0.1);
            color: #0284c7;
            border: 1px solid rgba(14, 165, 233, 0.2);
            transition: all 0.2s ease;
        }

        .btn-rincian:hover {
            background: #0284c7;
            color: white;
            border-color: #0284c7;
        }

        /* Action Buttons */
        .btn-group .btn {
            padding: 0.5rem;
            border-radius: 6px;
            font-size: 0.875rem;
            transition: all 0.2s ease;
        }

        .btn-group .btn:hover {
            transform: translateY(-1px);
        }

        .btn-eye {
            background: rgba(14, 165, 233, 0.1);
            color: #0284c7;
            border: 1px solid rgba(14, 165, 233, 0.2);
        }

        .btn-eye:hover {
            background: #0284c7;
            color: white;
        }

        /* Total Row Styling */
        .total-row td {
            background: #f8fafc !important;
            font-weight: 500;
            border-top: 2px solid #e2e8f0;
        }

        .total-amount {
            color: #0284c7;
            font-weight: 600;
            font-size: 1.125rem;
        }

        /* Status Indicators */
        .check-icon {
            color: #16a34a;
            font-size: 1rem;
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .table td, .table th {
                padding: 1rem !important;
            }

            .badge {
                padding: 0.375rem 0.625rem;
                font-size: 0.8125rem;
            }

            .maksud-text {
                font-size: 0.875rem;
            }
        }

        .action-bar {
            background: #fff;
            padding: 1rem;
            border-radius: 0.5rem;
            margin-bottom: 1.5rem;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
        }
        
        .action-bar .btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            font-weight: 500;
            transition: all 0.2s ease;
        }
        
        .action-bar .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        .action-bar .btn i {
            font-size: 1rem;
        }
        
        @media (max-width: 768px) {
            .action-bar {
                flex-direction: column;
                gap: 1rem;
            }
            
            .action-bar .d-flex {
                flex-direction: column;
                width: 100%;
            }
            
            .action-bar .btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
</head>
<body>
<div class="container-fluid mt-4">
    <?php if(isset($_SESSION['alert'])): ?>
    <div class="alert alert-<?= $_SESSION['alert']['type'] ?> alert-dismissible fade show" role="alert">
        <?= $_SESSION['alert']['message'] ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <?php 
    unset($_SESSION['alert']);
    endif; 
    ?>

    <!-- Modified Header -->
    <div class="header-wrapper">
        <div class="nav-breadcrumb">
            <a href="../dashboard.php">
                <i class="bi bi-house-door"></i>
                Dashboard
            </a>
            <span class="separator">/</span>
            <span class="text-white">Rincian Biaya</span>
            </div>
        <h1 class="page-title">Daftar Rincian Biaya Bimtek atau Sejenisnya</h1>
        <p class="page-subtitle">Daftar rincian biaya perjalanan dinas</p>

        <!-- Navigation Menu -->
        <div class="nav-menu">
            <a href="../dashboard.php" class="nav-link">
                <i class="bi bi-speedometer2"></i>
                Dashboard
            </a>
            <a href="../pegawai/pegawai.php" class="nav-link">
                <i class="bi bi-people-fill"></i>
                Pegawai
            </a>
            <a href="../surat_tugas/surat_tugas.php" class="nav-link">
                <i class="bi bi-file-earmark-text-fill"></i>
                Surat Tugas
            </a>
            <a href="../rincian_biaya/index.php" class="nav-link active">
                <i class="bi bi-cash-stack"></i>
                Rincian Biaya
            </a>
            <a href="../sppd_muka/sppd_muka.php" class="nav-link">
                <i class="bi bi-printer-fill"></i>
                Cetak SPPD
            </a>
            <a href="../rekap_sppd/index.php" class="nav-link">
                <i class="bi bi-file-text-fill"></i>
                Rekap SPPD
            </a>
            <a href="../laporan/laporan.php" class="nav-link">
                <i class="bi bi-journal-text"></i>
                Laporan
            </a>
        </div>
    </div>

    <!-- Action Bar with Back Button -->
    <div class="action-bar d-flex justify-content-between align-items-center">
    
        <div class="d-flex gap-2">
            <a href="../surat_tugas/lihat_surat.php?id=<?= $_GET['id'] ?? '' ?>" 
               class="btn btn-primary">
                <i class="bi bi-file-text"></i> 
                Lihat Surat Tugas
            </a>
            <a href="../rincian_biaya/index.php?id=<?= $_GET['id'] ?? '' ?>" 
               class="btn btn-success">
                <i class="bi bi-journal-text"></i> 
                Lihat Rincian Non-Bimtek
            </a>
        </div>
    </div>

            <!-- Table Card -->
            <div class="card">
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover" id="rincianTable">
                            <thead>
                                <tr>
                            <th class="text-center" style="width: 60px">No.</th>
                            <th style="width: 50%">Maksud</th>
                                    <th style="width: 10%">No. SPPD</th>
                                    <th style="width: 20%">Nama Pegawai</th>
                                    <th style="width: 20%" class="text-center">Total Biaya</th>
                            <th style="width: 15%" class="text-center">Aksi</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $no = 1;
                                $total_keseluruhan = 0;
                                while ($row = mysqli_fetch_assoc($result)): 
                                    // Get pegawai list and count
                                    $pegawai_list = explode('||', $row['pegawai_data']);
                                    $total_pegawai = count($pegawai_list);

                                    // Check if rincian exists
                                    $query_check = "SELECT COUNT(*) as count FROM rincian_bimtek_perorangan WHERE id_surat = ?";
                                    $stmt_check = mysqli_prepare($conn, $query_check);
                                    mysqli_stmt_bind_param($stmt_check, "i", $row['id_surat']);
                                    mysqli_stmt_execute($stmt_check);
                                    $result_check = mysqli_stmt_get_result($stmt_check);
                                    $check_data = mysqli_fetch_assoc($result_check);
                                    $has_rincian = $check_data['count'] > 0;
                                ?>
                                <tr class="main-row <?= (!$has_rincian) ? 'no-input-row' : '' ?>" data-row-id="<?= $row['id_surat'] ?>">
                                    <td class="text-center"><?= $no++ ?></td>
                                    <td>
                                        <div class="maksud-wrapper">
                                            <div class="jenis-surat-badge <?= strtolower(str_replace(' ', '-', $row['jenis_surat'])) ?>">
                                                <?php
                                                switch($row['jenis_surat']) {
                                                    case 'Dalam Daerah Dalam Provinsi':
                                                        echo '<i class="bi bi-geo-alt"></i> Dalam Daerah';
                                                        break;
                                                    case 'Luar Daerah Dalam Provinsi':
                                                        echo '<i class="bi bi-geo"></i> Luar Daerah';
                                                        break;
                                                    case 'Luar Daerah Luar Provinsi':
                                                        echo '<i class="bi bi-globe"></i> Luar Provinsi';
                                                        break;
                                                }
                                                ?>
                                            </div>
                                            <div class="maksud-text"><?= nl2br(htmlspecialchars($row['maksud'])) ?></div>
                                        </div>
                                    </td>
                                    <td><?= htmlspecialchars($row['no_sppd']) ?></td>
                                    <td>
                                        <div class="pegawai-wrapper">
                                            <div class="pegawai-list">
                                                <?php 
                                                foreach($pegawai_list as $index => $pegawai_info): 
                                                    if (empty($pegawai_info)) continue;
                                                    
                                                    list($id_pegawai, $nama_pegawai) = explode(':', $pegawai_info);
                                                    $hide_class = $index >= 2 ? 'pegawai-hidden d-none' : '';
                                                ?>
                                                <div class="pegawai-item <?= $hide_class ?>">
                                                    <div class="pegawai-name">
                                                        <span class="pegawai-number"><?= ($index + 1) ?></span>
                                                        <?= htmlspecialchars($nama_pegawai) ?>
                                                    </div>
                                                </div>
                                                <?php endforeach; ?>
                                            </div>
                                            <?php if($total_pegawai > 2): ?>
                                            <a href="#" class="lihat-semua" onclick="event.preventDefault(); togglePegawaiList(this)">
                                                <i class="bi bi-chevron-down"></i>
                                                <span class="toggle-text">Lihat Semua</span>
                                                <span class="toggle-count">(<?= $total_pegawai - 2 ?> lainnya)</span>
                                            </a>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td style="width: 20%" class="text-center">
                                        <div class="d-flex flex-column align-items-start">
                                            <?php
                                            // Get total components for this surat tugas
                                            $query_components = "SELECT 
                                                SUM(transport_dalam_provinsi) as total_transport_dalam,
                                                SUM(transport_luar_provinsi) as total_transport_luar,
                                                SUM(biaya_pesawat_berangkat) as total_pesawat_berangkat,
                                                SUM(biaya_pesawat_pulang) as total_pesawat_pulang,
                                                SUM(uang_harian_per_hari * 2) as total_uang_harian,
                                                SUM(uang_diklat_per_hari * (DATEDIFF(st.tanggal_kembali, st.tanggal_berangkat) - 1)) as total_uang_diklat,
                                                SUM(representatif_per_hari) as total_representatif_per_hari,
                                                SUM(representatif_per_hari * (DATEDIFF(st.tanggal_kembali, st.tanggal_berangkat) + 1)) as total_representatif,
                                                SUM(akomodasi) as total_akomodasi,
                                                SUM(biaya_jarak) as total_biaya_jarak
                                            FROM rincian_bimtek_perorangan r
                                            JOIN surat_tugas st ON r.id_surat = st.id_surat 
                                            WHERE r.id_surat = ?";
                                            
                                            $stmt_components = mysqli_prepare($conn, $query_components);
                                            mysqli_stmt_bind_param($stmt_components, "i", $row['id_surat']);
                                            mysqli_stmt_execute($stmt_components);
                                            $result_components = mysqli_stmt_get_result($stmt_components);
                                            $components = mysqli_fetch_assoc($result_components);

                                            $total_biaya = 0;
                                            if ($components) {
                                                if ($row['jenis_surat'] == 'Dalam Daerah Dalam Provinsi' || 
                                                    $row['jenis_surat'] == 'Luar Daerah Dalam Provinsi') {
                                                    
                                                    $transport = $components['total_transport_dalam'] ?? 0;
                                                    $uang_harian = $components['total_uang_harian'] ?? 0;
                                                    $uang_diklat = $components['total_uang_diklat'] ?? 0;
                                                    $representatif_per_hari = $components['total_representatif_per_hari'] ?? 0;
                                                    $total_representatif = $components['total_representatif'] ?? 0;
                                                    $akomodasi = $components['total_akomodasi'] ?? 0;
                                                    $biaya_jarak = $components['total_biaya_jarak'] ?? 0;
                                                    
                                                    // Calculate total
                                                    $total_biaya = $transport + $uang_harian + $uang_diklat + $total_representatif + $akomodasi + $biaya_jarak;
                                                    
                                                    // Urutkan tampilan: uang dulu, kemudian transport dan akomodasi
                                                    echo "<span class='badge bg-info mb-1'>Uang Harian: Rp " . number_format($uang_harian, 0, ',', '.') . "</span>";
                                                    echo "<span class='badge bg-info mb-1'>Uang Diklat: Rp " . number_format($uang_diklat, 0, ',', '.') . "</span>";
                                                    if ($representatif_per_hari > 0) {
                                                        echo "<span class='badge bg-info mb-1'>Uang representatif per hari: Rp " . number_format($representatif_per_hari, 0, ',', '.') . "</span>";
                                                        echo "<span class='badge bg-info mb-1'>Total Uang Representatif: Rp " . number_format($total_representatif, 0, ',', '.') . "</span>";
                                                    }
                                                    echo "<span class='badge bg-info mb-1'>Transport: Rp " . number_format($transport, 0, ',', '.') . "</span>";
                                                    echo "<span class='badge bg-info mb-1'>Akomodasi: Rp " . number_format($akomodasi, 0, ',', '.') . "</span>";
                                                    if ($biaya_jarak > 0) {
                                                        echo "<span class='badge bg-info mb-1'>Biaya Jarak: Rp " . number_format($biaya_jarak, 0, ',', '.') . "</span>";
                                                    }
                                                    
                                                } elseif ($row['jenis_surat'] == 'Luar Daerah Luar Provinsi') {
                                                    $transport_dalam = $components['total_transport_dalam'] ?? 0;
                                                    $transport_luar = $components['total_transport_luar'] ?? 0;
                                                    $pesawat_berangkat = $components['total_pesawat_berangkat'] ?? 0;
                                                    $pesawat_pulang = $components['total_pesawat_pulang'] ?? 0;
                                                    $uang_harian = $components['total_uang_harian'] ?? 0;
                                                    $uang_diklat = $components['total_uang_diklat'] ?? 0;
                                                    $representatif_per_hari = $components['total_representatif_per_hari'] ?? 0;
                                                    $total_representatif = $components['total_representatif'] ?? 0;
                                                    $akomodasi = $components['total_akomodasi'] ?? 0;
                                                    
                                                    // Calculate total for luar provinsi
                                                    $total_biaya = $transport_dalam + $transport_luar + $pesawat_berangkat + 
                                                                  $pesawat_pulang + $uang_harian + $uang_diklat + $total_representatif + $akomodasi;
                                                    
                                                    // Urutkan tampilan: uang dulu, kemudian transport
                                                    echo "<span class='badge bg-info mb-1'>Uang Harian: Rp " . number_format($uang_harian, 0, ',', '.') . "</span>";
                                                    echo "<span class='badge bg-info mb-1'>Uang Diklat: Rp " . number_format($uang_diklat, 0, ',', '.') . "</span>";
                                                    if ($representatif_per_hari > 0) {
                                                        echo "<span class='badge bg-info mb-1'>Uang representatif per hari: Rp " . number_format($representatif_per_hari, 0, ',', '.') . "</span>";
                                                        echo "<span class='badge bg-info mb-1'>Total Uang Representatif: Rp " . number_format($total_representatif, 0, ',', '.') . "</span>";
                                                    }
                                                    echo "<span class='badge bg-info mb-1'>Transport Dalam: Rp " . number_format($transport_dalam, 0, ',', '.') . "</span>";
                                                    echo "<span class='badge bg-info mb-1'>Transport Luar: Rp " . number_format($transport_luar, 0, ',', '.') . "</span>";
                                                    echo "<span class='badge bg-info mb-1'>Pesawat Berangkat: Rp " . number_format($pesawat_berangkat, 0, ',', '.') . "</span>";
                                                    echo "<span class='badge bg-info mb-1'>Pesawat Pulang: Rp " . number_format($pesawat_pulang, 0, ',', '.') . "</span>";
                                                    echo "<span class='badge bg-info mb-1'>Akomodasi: Rp " . number_format($akomodasi, 0, ',', '.') . "</span>";
                                                }
                                            }
                                            
                                            // Add to total keseluruhan
                                            $total_keseluruhan += $total_biaya;

                                            if ($total_biaya > 0) {
                                                echo "<span class='badge bg-success'>Total Biaya: Rp " . number_format($total_biaya, 0, ',', '.') . "</span>";
                                            } else {
                                                echo "<span class='badge bg-warning'>Belum ada rincian biaya</span>";
                                            }
                                            ?>
                                        </div>
                                    </td>
                                    <td class="text-center">
                                        <div class="btn-group">
                                            <a href="detail_rincian.php?id=<?= $row['id_surat'] ?>" 
                                               class="btn btn-sm btn-info" title="Detail">
                                                <i class="bi bi-eye"></i>
                                            </a>
                                            <a href="cetak_tanda_terima_khusus.php?id_surat=<?= $row['id_surat'] ?>" 
                                               class="btn btn-sm btn-primary" title="Cetak Tanda Terima">
                                                <i class="bi bi-printer"></i>
                                            </a>
                                            <div class="btn-group">
                                                <button type="button" class="btn btn-sm btn-secondary dropdown-toggle" data-bs-toggle="dropdown">
                                                    <i class="bi bi-three-dots"></i>
                                                </button>
                                                <ul class="dropdown-menu dropdown-menu-end">
                                                    <li>
                                                        <a class="dropdown-item" href="cetak_rincian_khusus.php?id_surat=<?= $row['id_surat'] ?>&id_pegawai=<?= explode(':', $pegawai_list[0])[0] ?>">
                                                            <i class="bi bi-printer"></i> Cetak Rincian Biaya
                                                        </a>
                                                    </li>
                                                    <li>
                                                        <a class="dropdown-item" href="cetak_rincian_riil_khusus.php?id_surat=<?= $row['id_surat'] ?>&id_pegawai=<?= explode(':', $pegawai_list[0])[0] ?>">
                                                            <i class="bi bi-printer"></i> Cetak Rincian Riil
                                                        </a>
                                                    </li>
                                                </ul>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                            <tfoot>
                                <tr class="total-row">
                                    <td colspan="4" class="text-end pe-3">
                                        <div><strong>Total Keseluruhan:</strong></div>
                                        <div class="text-muted small">Terbilang:</div>
                                    </td>
                                    <td colspan="2" class="text-end pe-3">
                                        <div><strong class="total-amoun">Rp <?= number_format($total_keseluruhan, 0, ',', '.') ?></strong></div>
                                        <div class="text-muted small fst-italic"><?= ucfirst(terbilang($total_keseluruhan)) ?></div>
                                    </td>
                                </tr>
                            </tfoot>
                        </table>
            </div>
        </div>
    </div>
</div>

<!-- Scripts -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/dataTables.responsive.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/responsive.bootstrap5.min.js"></script>

<script>
$(document).ready(function() {
    // Inisialisasi event handler untuk tombol lihat semua
    $(document).on('click', '.lihat-semua', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        const wrapper = $(this).closest('.pegawai-wrapper');
        const hiddenItems = wrapper.find('.pegawai-hidden');
        const toggleText = wrapper.find('.toggle-text');
        const toggleCount = wrapper.find('.toggle-count');
        const icon = wrapper.find('.bi-chevron-down, .bi-chevron-up');
        
        if (wrapper.hasClass('expanded')) {
            // Collapse
            wrapper.removeClass('expanded');
            toggleText.text('Lihat Semua');
            toggleCount.show();
            hiddenItems.addClass('d-none');
            icon.removeClass('bi-chevron-up').addClass('bi-chevron-down');
        } else {
            // Expand
            wrapper.addClass('expanded');
            toggleText.text('Sembunyikan');
            toggleCount.hide();
            hiddenItems.removeClass('d-none');
            icon.removeClass('bi-chevron-down').addClass('bi-chevron-up');
        }
    });
});

function hapusRincian(id) {
    if(confirm('Apakah Anda yakin ingin menghapus rincian biaya ini?')) {
        fetch('hapus_rincian.php?id=' + id)
            .then(response => response.json())
            .then(data => {
                if(data.success) {
                    alert('Data berhasil dihapus');
                    location.reload();
                } else {
                    alert('Gagal menghapus data: ' + data.message);
                }
            })
            .catch(error => {
                alert('Terjadi kesalahan saat menghapus data');
                console.error('Error:', error);
            });
    }
}
</script>

</body>
</html>