<?php
session_start();
require_once '../config/database.php';
require_once '../config/config.php';

if (isset($_GET['id'])) {
    $id_surat = $_GET['id'];
    
    // Query untuk mengambil data surat tugas dan semua pegawai terkait
    $query_surat = "SELECT st.*, 
                    GROUP_CONCAT(DISTINCT p.id_pegawai) as pegawai_ids,
                    GROUP_CONCAT(DISTINCT p.nama_pegawai SEPARATOR ', ') as nama_pegawai,
                    GROUP_CONCAT(DISTINCT p.nama_pegawai) as nama_pegawai_list,
                    GROUP_CONCAT(DISTINCT p.jabatan) as jabatan_pegawai,
                    st.tujuan,
                    st.tempat_kegiatan,
                    DATEDIFF(st.tanggal_kembali, st.tanggal_berangkat) + 1 as lama_perjalanan
                    FROM surat_tugas st 
                    LEFT JOIN detail_surat_tugas dst ON st.id_surat = dst.id_surat
                    LEFT JOIN pegawai p ON dst.id_pegawai = p.id_pegawai 
                    WHERE st.id_surat = ?
                    GROUP BY st.id_surat";
    
    $stmt = mysqli_prepare($conn, $query_surat);
    mysqli_stmt_bind_param($stmt, "i", $id_surat);
    mysqli_stmt_execute($stmt);
    $result_surat = mysqli_stmt_get_result($stmt);
    $surat = mysqli_fetch_assoc($result_surat);

    if ($surat) {
        // Mendapatkan array dari id pegawai dan nama pegawai
        $pegawai_ids = explode(',', $surat['pegawai_ids']);
        $nama_pegawai_list = explode(',', $surat['nama_pegawai_list']);
        $jabatan_pegawai = explode(',', $surat['jabatan_pegawai']);
        
        // Fungsi untuk mengecek apakah ada pejabat tinggi
        function hasPejabatTinggi($jabatan_array) {
            $pejabat_tinggi = ['Kepala Dinas', 'Kepala Badan', 'Ketua DPR', 'Wakil DPR', 'Bupati', 'Sekretaris'];
            foreach ($jabatan_array as $jabatan) {
                if (in_array(trim($jabatan), $pejabat_tinggi)) {
                    return true;
                }
            }
            return false;
        }
        
        $show_representatif = hasPejabatTinggi($jabatan_pegawai);
        
        // Query untuk mengambil rincian biaya yang sudah ada
        $query_rincian = "SELECT * FROM rincian_bimtek_perorangan WHERE id_surat = ? LIMIT 1";
        $stmt = mysqli_prepare($conn, $query_rincian);
        mysqli_stmt_bind_param($stmt, "i", $id_surat);
        mysqli_stmt_execute($stmt);
        $result_rincian = mysqli_stmt_get_result($stmt);
        $rincian = mysqli_fetch_assoc($result_rincian);
    }
}

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $id_surat = $_POST['id_surat'];
    
    // Ambil semua id_pegawai dari detail_surat_tugas
    $query_pegawai = "SELECT id_pegawai FROM detail_surat_tugas WHERE id_surat = ?";
    $stmt = mysqli_prepare($conn, $query_pegawai);
    mysqli_stmt_bind_param($stmt, "i", $id_surat);
    mysqli_stmt_execute($stmt);
    $result_pegawai = mysqli_stmt_get_result($stmt);
    
    if (mysqli_num_rows($result_pegawai) == 0) {
        $_SESSION['alert'] = [
            'type' => 'error',
            'message' => 'Data pegawai tidak ditemukan'
        ];
        header("Location: index.php");
        exit();
    }
    
    // Mengambil dan membersihkan input
    $uang_harian_per_hari = str_replace('.', '', $_POST['uang_harian_per_hari']);
    $total_uang_harian = $uang_harian_per_hari * 2; // Hanya 2 hari (hari pertama dan terakhir)
    
    $representatif_per_hari = isset($_POST['representatif']) ? str_replace('.', '', $_POST['representatif']) : 0;
    $total_representatif = $representatif_per_hari * $surat['lama_perjalanan'];
    
    $uang_diklat_per_hari = isset($_POST['uang_diklat']) ? str_replace('.', '', $_POST['uang_diklat']) : 0;
    $sisa_hari_diklat = $surat['lama_perjalanan'] - 2;
    $total_uang_diklat = $uang_diklat_per_hari * ($sisa_hari_diklat > 0 ? $sisa_hari_diklat : 0);
    
    $transport_dalam_provinsi = isset($_POST['transport_dalam']) ? str_replace('.', '', $_POST['transport_dalam']) : 0;
    $transport_luar_provinsi = isset($_POST['transport_luar']) ? str_replace('.', '', $_POST['transport_luar']) : 0;
    $biaya_pesawat_berangkat = isset($_POST['pesawat_berangkat']) ? str_replace('.', '', $_POST['pesawat_berangkat']) : 0;
    $biaya_pesawat_pulang = isset($_POST['pesawat_kembali']) ? str_replace('.', '', $_POST['pesawat_kembali']) : 0;
    $akomodasi = isset($_POST['akomodasi']) ? str_replace('.', '', $_POST['akomodasi']) : 0;
    $jarak = isset($_POST['jarak']) ? $_POST['jarak'] : 0;
    $biaya_jarak = isset($_POST['biaya_jarak']) ? str_replace('.', '', $_POST['biaya_jarak']) : 0;
    
    // Hitung total biaya
    $total_biaya = $total_uang_harian + $total_representatif + $total_uang_diklat + $transport_dalam_provinsi + 
                   $transport_luar_provinsi + $biaya_pesawat_berangkat + 
                   $biaya_pesawat_pulang + $akomodasi + $biaya_jarak;

    $success = true;
    $error_messages = [];

    // Loop untuk setiap pegawai
    while ($row_pegawai = mysqli_fetch_assoc($result_pegawai)) {
        $id_pegawai = $row_pegawai['id_pegawai'];
        
        // Ambil nama pegawai
        $query_nama = "SELECT nama_pegawai FROM pegawai WHERE id_pegawai = ?";
        $stmt_nama = mysqli_prepare($conn, $query_nama);
        mysqli_stmt_bind_param($stmt_nama, "i", $id_pegawai);
        mysqli_stmt_execute($stmt_nama);
        $result_nama = mysqli_stmt_get_result($stmt_nama);
        $row_nama = mysqli_fetch_assoc($result_nama);
        $nama_pegawai = $row_nama['nama_pegawai'];
        
        // Cek apakah sudah ada rincian biaya untuk pegawai ini
        $check_query = "SELECT id_rincian_perorangan FROM rincian_bimtek_perorangan WHERE id_surat = ? AND id_pegawai = ?";
        $stmt = mysqli_prepare($conn, $check_query);
        mysqli_stmt_bind_param($stmt, "ii", $id_surat, $id_pegawai);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);

        if (mysqli_num_rows($result) > 0) {
            // Update data yang ada
            $row = mysqli_fetch_assoc($result);
            $id_rincian = $row['id_rincian_perorangan'];
            
            $query = "UPDATE rincian_bimtek_perorangan SET 
                      uang_harian_per_hari = ?,
                      total_uang_harian = ?,
                      representatif_per_hari = ?,
                      total_representatif = ?,
                      uang_diklat_per_hari = ?,
                      total_uang_diklat = ?,
                      transport_dalam_provinsi = ?,
                      transport_luar_provinsi = ?,
                      biaya_pesawat_berangkat = ?,
                      biaya_pesawat_pulang = ?,
                      akomodasi = ?,
                      total_biaya = ?,
                      updated_at = CURRENT_TIMESTAMP
                      WHERE id_rincian_perorangan = ?";
                      
            $stmt = mysqli_prepare($conn, $query);
            mysqli_stmt_bind_param($stmt, "ddddddddddddi", 
                $uang_harian_per_hari,
                $total_uang_harian,
                $representatif_per_hari,
                $total_representatif,
                $uang_diklat_per_hari,
                $total_uang_diklat,
                $transport_dalam_provinsi,
                $transport_luar_provinsi,
                $biaya_pesawat_berangkat,
                $biaya_pesawat_pulang,
                $akomodasi,
                $total_biaya,
                $id_rincian
            );
        } else {
            // Insert data baru
            $query = "INSERT INTO rincian_bimtek_perorangan (
                        id_surat,
                        id_pegawai,
                        nama_pegawai,
                        uang_harian_per_hari,
                        total_uang_harian,
                        representatif_per_hari,
                        total_representatif,
                        uang_diklat_per_hari,
                        total_uang_diklat,
                        transport_dalam_provinsi,
                        transport_luar_provinsi,
                        biaya_pesawat_berangkat,
                        biaya_pesawat_pulang,
                        akomodasi,
                        jarak,
                        biaya_jarak,
                        total_biaya,
                        created_at,
                        updated_at
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)";
                    
            $stmt = mysqli_prepare($conn, $query);
            mysqli_stmt_bind_param($stmt, "iisddddddddddiddd", 
                $id_surat,
                $id_pegawai,
                $nama_pegawai,
                $uang_harian_per_hari,
                $total_uang_harian,
                $representatif_per_hari,
                $total_representatif,
                $uang_diklat_per_hari,
                $total_uang_diklat,
                $transport_dalam_provinsi,
                $transport_luar_provinsi,
                $biaya_pesawat_berangkat,
                $biaya_pesawat_pulang,
                $akomodasi,
                $jarak,
                $biaya_jarak,
                $total_biaya
            );
        }

        if (!mysqli_stmt_execute($stmt)) {
            $success = false;
            $error_messages[] = "Gagal menyimpan rincian biaya untuk pegawai: $nama_pegawai - " . mysqli_error($conn);
        }
    }

    if ($success) {
        $_SESSION['alert'] = [
            'type' => 'success',
            'message' => 'Rincian biaya berhasil disimpan untuk semua pegawai'
        ];
        header("Location: index.php");
        exit();
    } else {
        $_SESSION['alert'] = [
            'type' => 'error',
            'message' => 'Terjadi kesalahan: ' . implode(", ", $error_messages)
        ];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Tambah/Edit Rincian Biaya - SPPD</title>
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyNCAyNCI+PHBhdGggZmlsbD0iIzRlNzNkZiIgZD0iTTIwLDhINFY2aDIwdjJIMjB6IE0xNSwxMEg0djJoMTFWMTB6IE00LDE2aDd2Mkg0VjE2eiBNMjEuNywxMy4zTDIwLjMsMTJsMS40LTEuNEMyMi4xLDEwLjIsMjIuMSw5LjgsMjEuNyw5LjRsLTEuMS0xLjFDMjAuMiw3LjksMTkuOCw3LjksMTkuNCw4LjNMMTgsMTBsLTEuNC0xLjRjLTAuNC0wLjQtMS0wLjQtMS40LDBsLTcuNSw3LjVjLTAuMiwwLjItMC4zLDAuNC0wLjMsMC43djIuOWMwLDAuNSwwLjQsMC45LDAuOSwwLjloMi45YzAuMywwLDAuNS0wLjEsMC43LTAuM2w3LjUtNy41QzIyLjEsMTQuMywyMi4xLDEzLjcsMjEuNywxMy4zeiIvPjwvc3ZnPg==">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #f6f9fc 0%, #eef2f7 100%);
            min-height: 100vh;
            margin: 0;
            padding: 0;
            font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
        }

        .header-wrapper {
            background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
            padding: 1.5rem 2rem;
            margin: 0;
            color: white;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        }

        .content-wrapper {
            padding: 2rem;
        }

        .card {
            background: rgba(255, 255, 255, 0.95);
            border: none;
            border-radius: 12px;
            box-shadow: 0 4px 25px rgba(0, 0, 0, 0.05);
            backdrop-filter: blur(10px);
            margin-bottom: 2rem;
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }

        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.1);
        }

        .card-header {
            background: transparent;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
            padding: 1.5rem;
        }

        .card-header h5 {
            margin: 0;
            font-weight: 600;
            color: #2c3e50;
        }

        .card-body {
            padding: 1.5rem;
        }

        .table {
            margin: 0;
        }

        .table th {
            font-weight: 600;
            color: #4a5568;
            padding: 1rem;
            border: none;
        }

        .table td {
            padding: 1rem;
            border: none;
            color: #2d3748;
        }

        .form-label {
            font-weight: 500;
            color: #4a5568;
            margin-bottom: 0.5rem;
        }

        .form-control {
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 0.75rem 1rem;
            transition: all 0.2s ease;
        }

        .form-control:focus {
            border-color: #4e73df;
            box-shadow: 0 0 0 3px rgba(78, 115, 223, 0.1);
        }

        .input-group {
            border-radius: 8px;
            overflow: hidden;
        }

        .input-group-text {
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            color: #4a5568;
            padding: 0.75rem 1rem;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            font-weight: 500;
            transition: all 0.2s ease;
        }

        .btn-primary {
            background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
            border: none;
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, #224abe 0%, #1a3891 100%);
            transform: translateY(-2px);
        }

        .btn-secondary {
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            color: #4a5568;
        }

        .btn-secondary:hover {
            background: #edf2f7;
            transform: translateY(-2px);
        }

        .alert {
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1rem;
            border: none;
        }

        .alert-info {
            background: rgba(66, 153, 225, 0.1);
            color: #2b6cb0;
        }

        .breadcrumb {
            background: transparent;
            padding: 0;
            margin: 0.5rem 0 0;
        }

        .breadcrumb-item a {
            color: rgba(255, 255, 255, 0.8);
            text-decoration: none;
            transition: color 0.2s ease;
        }

        .breadcrumb-item a:hover {
            color: white;
        }

        .breadcrumb-item.active {
            color: white;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .content-wrapper {
                padding: 1rem;
            }

            .card {
                margin-bottom: 1rem;
            }

            .header-wrapper {
                padding: 1rem;
            }

            .table-responsive {
                margin: 0 -1rem;
                width: calc(100% + 2rem);
            }
        }

        /* Custom Scrollbar */
        ::-webkit-scrollbar {
            width: 8px;
            height: 8px;
        }

        ::-webkit-scrollbar-track {
            background: #f1f5f9;
        }

        ::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 4px;
        }

        ::-webkit-scrollbar-thumb:hover {
            background: #94a3b8;
        }
    </style>
</head>
<body>
    <div class="header-wrapper">
        <h2 class="mb-0">Tambah/Edit Rincian Biaya</h2>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item"><a href="../dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item"><a href="index.php">Rincian Biaya</a></li>
                <li class="breadcrumb-item active">Tambah/Edit Rincian</li>
            </ol>
        </nav>
    </div>

    <div class="content-wrapper">
        <?php if (isset($surat)): ?>
        <div class="card">
            <div class="card-header">
                <h5 class="card-title">Informasi Surat Tugas</h5>
            </div>
            <div class="card-body">
                <div class="row g-4">
                    <div class="col-md-6">
                        <div class="info-group">
                            <label class="form-label">ID Surat</label>
                            <p class="mb-3"><?= htmlspecialchars($surat['id_surat'], ENT_QUOTES, 'UTF-8') ?></p>
                        </div>
                        <div class="info-group">
                            <label class="form-label">Jenis Surat</label>
                            <p class="mb-3"><?= htmlspecialchars($surat['jenis_surat'], ENT_QUOTES, 'UTF-8') ?></p>
                        </div>
                        <div class="info-group">
                            <label class="form-label">No. Surat</label>
                            <p class="mb-3"><?= htmlspecialchars($surat['no_surat'], ENT_QUOTES, 'UTF-8') ?></p>
                        </div>
                        <div class="info-group">
                            <label class="form-label">No. SPPD</label>
                            <p class="mb-3"><?= htmlspecialchars($surat['no_sppd'], ENT_QUOTES, 'UTF-8') ?></p>
                        </div>
                        <div class="info-group">
                            <label class="form-label">Nama Pegawai</label>
                            <p class="mb-3"><?= htmlspecialchars($surat['nama_pegawai'], ENT_QUOTES, 'UTF-8') ?></p>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="info-group">
                            <label class="form-label">Tanggal Berangkat</label>
                            <p class="mb-3"><?= date('d/m/Y', strtotime($surat['tanggal_berangkat'])) ?></p>
                        </div>
                        <div class="info-group">
                            <label class="form-label">Tanggal Kembali</label>
                            <p class="mb-3"><?= date('d/m/Y', strtotime($surat['tanggal_kembali'])) ?></p>
                        </div>
                        <div class="info-group">
                            <label class="form-label">Tujuan</label>
                            <p class="mb-3"><?= htmlspecialchars($surat['tujuan'], ENT_QUOTES, 'UTF-8') ?></p>
                        </div>
                        <div class="info-group">
                            <label class="form-label">Tempat Kegiatan</label>
                            <p class="mb-3"><?= !empty($surat['tempat_kegiatan']) ? htmlspecialchars($surat['tempat_kegiatan'], ENT_QUOTES, 'UTF-8') : '-' ?></p>
                        </div>
                        <div class="info-group">
                            <label class="form-label">Lama Perjalanan</label>
                            <p class="mb-3"><?= $surat['lama_perjalanan'] ?> Hari</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h5 class="card-title">Form Rincian Biaya</h5>
            </div>
            <div class="card-body">
                <form method="POST" action="">
                    <input type="hidden" name="id_surat" value="<?= $id_surat ?>">
                    <input type="hidden" name="id_pegawai" value="<?= $id_pegawai ?>">
                    
                    <?php
                    // Decode data rincian yang sudah ada jika ada
                    $biaya_detail = isset($rincian['biaya_detail']) ? json_decode($rincian['biaya_detail'], true) : [];
                    
                    // Form dinamis berdasarkan jenis surat
                    if ($surat['jenis_surat'] == 'Dalam Daerah Dalam Provinsi'):
                    ?>
                        <!-- Grup Uang Harian dan Representatif -->
                        <div class="card mb-4">
                            <div class="card-header bg-light">
                                <h6 class="mb-0">Uang Harian <?= $show_representatif ? '& Representatif' : '' ?></h6>
                            </div>
                            <div class="card-body">
                                <div class="row g-3">
                                    <div class="col-md-<?= $show_representatif ? '6' : '12' ?>">
                                        <label class="form-label">Uang Harian (2 Hari)</label>
                                        <div class="input-group">
                                            <span class="input-group-text">Rp</span>
                                            <input type="text" class="form-control money" name="uang_harian_per_hari" id="uang_harian_per_hari" 
                                                   value="<?= isset($rincian['uang_harian_per_hari']) ? number_format($rincian['uang_harian_per_hari'], 0, ',', '.') : '0' ?>"
                                                   onkeyup="hitungTotalUangHarian()" required>
                                        </div>
                                        <div class="mt-2">
                                            <small class="text-muted">Hanya dihitung untuk hari pertama dan terakhir</small>
                                            <div id="perhitungan_uang_harian" class="alert alert-info mt-2" style="display:none;">
                                                Perhitungan: <span id="rincian_uang_harian"></span>
                                                <br>
                                                Total: <strong><span id="total_uang_harian"></span></strong>
                                            </div>
                                        </div>
                                    </div>
                                    <?php if ($show_representatif): ?>
                                    <div class="col-md-6">
                                        <label class="form-label">Representatif (per hari)</label>
                                        <div class="input-group">
                                            <span class="input-group-text">Rp</span>
                                            <input type="text" class="form-control money" name="representatif_per_hari" id="representatif_per_hari"
                                                   value="<?= isset($rincian['representatif']) ? number_format($rincian['representatif']/$surat['lama_perjalanan'], 0, ',', '.') : '0' ?>" 
                                                   onkeyup="hitungTotalRepresentatif()" required>
                                        </div>
                                        <div class="mt-2">
                                            <small class="text-muted">Khusus untuk pejabat tinggi, dihitung per hari</small>
                                            <div id="perhitungan_representatif" class="alert alert-info mt-2" style="display:none;">
                                                Perhitungan: <span id="rincian_representatif"></span>
                                                <br>
                                                Total: <strong><span id="total_representatif"></span></strong>
                                            </div>
                                        </div>
                                        <input type="hidden" name="representatif" id="representatif_total" 
                                               value="<?= isset($rincian['representatif']) ? $rincian['representatif'] : '0' ?>">
                                    </div>
                                    <?php else: ?>
                                        <input type="hidden" name="representatif" value="0">
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Grup Uang Diklat -->
                        <div class="card mb-4">
                            <div class="card-header bg-light">
                                <h6 class="mb-0">Uang Diklat</h6>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-12">
                                        <label class="form-label">Uang Diklat (<?= max(0, $surat['lama_perjalanan'] - 2) ?> Hari)</label>
                                        <div class="input-group">
                                            <span class="input-group-text">Rp</span>
                                            <input type="text" class="form-control money" name="uang_diklat" id="uang_diklat_per_hari"
                                                   value="<?= isset($rincian['uang_diklat_per_hari']) ? number_format($rincian['uang_diklat_per_hari'], 0, ',', '.') : '0' ?>"
                                                   onkeyup="hitungTotalUangDiklat()" required>
                                        </div>
                                        <div class="mt-2">
                                            <small class="text-muted">Dihitung untuk sisa hari (total hari - 2 hari uang harian)</small>
                                            <div id="perhitungan_uang_diklat" class="alert alert-info mt-2" style="display:none;">
                                                Perhitungan: <span id="rincian_uang_diklat"></span>
                                                <br>
                                                Total: <strong><span id="total_uang_diklat"></span></strong>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Grup Transport & Jarak -->
                        <div class="card mb-4">
                            <div class="card-header bg-light">
                                <h6 class="mb-0">Transport & Jarak</h6>
                            </div>
                            <div class="card-body">
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Transport</label>
                                        <div class="input-group">
                                            <span class="input-group-text">Rp</span>
                                            <input type="text" class="form-control money" name="transport_dalam" 
                                                   value="<?= isset($rincian['transport_dalam_provinsi']) ? number_format($rincian['transport_dalam_provinsi'], 0, ',', '.') : '0' ?>" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Jarak (KM)</label>
                                        <input type="number" class="form-control" id="jarak" name="jarak" 
                                               value="<?= isset($rincian['jarak']) ? $rincian['jarak'] : '' ?>" 
                                               onkeyup="hitungBiayaJarak()" onchange="hitungBiayaJarak()">
                                        <small class="text-muted">Biaya per kilometer: Rp 5.000</small>
                                    </div>
                                    <div class="col-12">
                                        <label class="form-label">Biaya Jarak</label>
                                        <div class="input-group">
                                            <span class="input-group-text">Rp</span>
                                            <input type="text" class="form-control money" id="biaya_jarak" name="biaya_jarak" 
                                                   value="<?= isset($rincian['biaya_jarak']) ? number_format($rincian['biaya_jarak'], 0, ',', '.') : '0' ?>" readonly>
                                        </div>
                                        <div id="perhitungan_biaya_jarak" class="alert alert-info mt-2" style="display:none;">
                                            Perhitungan: <span id="rincian_biaya_jarak"></span>
                                            <br>
                                            Total: <strong><span id="total_biaya_jarak"></span></strong>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Grup Akomodasi -->
                        <div class="card mb-4">
                            <div class="card-header bg-light">
                                <h6 class="mb-0">Akomodasi</h6>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-12">
                                        <label class="form-label">Biaya Akomodasi</label>
                                        <div class="input-group">
                                            <span class="input-group-text">Rp</span>
                                            <input type="text" class="form-control money" name="akomodasi" 
                                                   value="<?= isset($rincian['akomodasi']) ? number_format($rincian['akomodasi'], 0, ',', '.') : '0' ?>" required>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                    <?php elseif ($surat['jenis_surat'] == 'Luar Daerah Dalam Provinsi'): ?>
                        <div class="mb-3">
                            <label class="form-label">Uang Harian (2 Hari)</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="uang_harian_per_hari" id="uang_harian_per_hari" 
                                       value="<?= isset($rincian['uang_harian_per_hari']) ? number_format($rincian['uang_harian_per_hari'], 0, ',', '.') : '0' ?>"
                                       onkeyup="hitungTotalUangHarian()" required>
                            </div>
                            <div class="mt-2">
                                <small class="text-muted">Hanya dihitung untuk hari pertama dan terakhir</small>
                                <div id="perhitungan_uang_harian" class="alert alert-info mt-2" style="display:none;">
                                    Perhitungan: <span id="rincian_uang_harian"></span>
                                    <br>
                                    Total: <strong><span id="total_uang_harian"></span></strong>
                                </div>
                            </div>
                        </div>

                        <?php if ($show_representatif): ?>
                        <div class="mb-3">
                            <label class="form-label">Representatif (per hari)</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="representatif_per_hari" id="representatif_per_hari"
                                       value="<?= isset($rincian['representatif']) ? number_format($rincian['representatif']/$surat['lama_perjalanan'], 0, ',', '.') : '0' ?>" 
                                       onkeyup="hitungTotalRepresentatif()" required>
                            </div>
                            <div class="mt-2">
                                <small class="text-muted">Khusus untuk pejabat tinggi, dihitung per hari</small>
                                <div id="perhitungan_representatif" class="alert alert-info mt-2" style="display:none;">
                                    Perhitungan: <span id="rincian_representatif"></span>
                                    <br>
                                    Total: <strong><span id="total_representatif"></span></strong>
                                </div>
                            </div>
                            <input type="hidden" name="representatif" id="representatif_total" 
                                   value="<?= isset($rincian['representatif']) ? $rincian['representatif'] : '0' ?>">
                        </div>
                        <?php else: ?>
                            <input type="hidden" name="representatif" value="0">
                        <?php endif; ?>

                        <div class="mb-3">
                            <label class="form-label">Uang Diklat (<?= max(0, $surat['lama_perjalanan'] - 2) ?> Hari)</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="uang_diklat" id="uang_diklat_per_hari"
                                       value="<?= isset($rincian['uang_diklat_per_hari']) ? number_format($rincian['uang_diklat_per_hari'], 0, ',', '.') : '0' ?>"
                                       onkeyup="hitungTotalUangDiklat()" required>
                            </div>
                            <div class="mt-2">
                                <small class="text-muted">Dihitung untuk sisa hari (total hari - 2 hari uang harian)</small>
                                <div id="perhitungan_uang_diklat" class="alert alert-info mt-2" style="display:none;">
                                    Perhitungan: <span id="rincian_uang_diklat"></span>
                                    <br>
                                    Total: <strong><span id="total_uang_diklat"></span></strong>
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Transport</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="transport_dalam" 
                                       value="<?= isset($rincian['transport_dalam_provinsi']) ? number_format($rincian['transport_dalam_provinsi'], 0, ',', '.') : '0' ?>" required>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Akomodasi</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="akomodasi" 
                                       value="<?= isset($rincian['akomodasi']) ? number_format($rincian['akomodasi'], 0, ',', '.') : '0' ?>" required>
                            </div>
                        </div>

                    <?php else: // Luar Daerah Luar Provinsi ?>
                        <div class="mb-3">
                            <label class="form-label">Uang Harian (2 Hari)</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="uang_harian_per_hari" id="uang_harian_per_hari" 
                                       value="<?= isset($rincian['uang_harian_per_hari']) ? number_format($rincian['uang_harian_per_hari'], 0, ',', '.') : '0' ?>"
                                       onkeyup="hitungTotalUangHarian()" required>
                            </div>
                            <div class="mt-2">
                                <small class="text-muted">Hanya dihitung untuk hari pertama dan terakhir</small>
                                <div id="perhitungan_uang_harian" class="alert alert-info mt-2" style="display:none;">
                                    Perhitungan: <span id="rincian_uang_harian"></span>
                                    <br>
                                    Total: <strong><span id="total_uang_harian"></span></strong>
                                </div>
                            </div>
                        </div>

                        <?php if ($show_representatif): ?>
                        <div class="mb-3">
                            <label class="form-label">Representatif (per hari)</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="representatif_per_hari" id="representatif_per_hari"
                                       value="<?= isset($rincian['representatif']) ? number_format($rincian['representatif']/$surat['lama_perjalanan'], 0, ',', '.') : '0' ?>" 
                                       onkeyup="hitungTotalRepresentatif()" required>
                            </div>
                            <div class="mt-2">
                                <small class="text-muted">Khusus untuk pejabat tinggi, dihitung per hari</small>
                                <div id="perhitungan_representatif" class="alert alert-info mt-2" style="display:none;">
                                    Perhitungan: <span id="rincian_representatif"></span>
                                    <br>
                                    Total: <strong><span id="total_representatif"></span></strong>
                                </div>
                            </div>
                            <input type="hidden" name="representatif" id="representatif_total" 
                                   value="<?= isset($rincian['representatif']) ? $rincian['representatif'] : '0' ?>">
                        </div>
                        <?php else: ?>
                            <input type="hidden" name="representatif" value="0">
                        <?php endif; ?>

                        <div class="mb-3">
                            <label class="form-label">Uang Diklat (<?= max(0, $surat['lama_perjalanan'] - 2) ?> Hari)</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="uang_diklat" id="uang_diklat_per_hari"
                                       value="<?= isset($rincian['uang_diklat_per_hari']) ? number_format($rincian['uang_diklat_per_hari'], 0, ',', '.') : '0' ?>"
                                       onkeyup="hitungTotalUangDiklat()" required>
                            </div>
                            <div class="mt-2">
                                <small class="text-muted">Dihitung untuk sisa hari (total hari - 2 hari uang harian)</small>
                                <div id="perhitungan_uang_diklat" class="alert alert-info mt-2" style="display:none;">
                                    Perhitungan: <span id="rincian_uang_diklat"></span>
                                    <br>
                                    Total: <strong><span id="total_uang_diklat"></span></strong>
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Transport Dalam Provinsi</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="transport_dalam" 
                                       value="<?= isset($rincian['transport_dalam_provinsi']) ? number_format($rincian['transport_dalam_provinsi'], 0, ',', '.') : '0' ?>" required>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Transport Luar Provinsi</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="transport_luar" 
                                       value="<?= isset($rincian['transport_luar_provinsi']) ? number_format($rincian['transport_luar_provinsi'], 0, ',', '.') : '0' ?>" required>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Akomodasi</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="akomodasi" 
                                       value="<?= isset($rincian['akomodasi']) ? number_format($rincian['akomodasi'], 0, ',', '.') : '0' ?>" required>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Biaya Pesawat Berangkat</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="pesawat_berangkat" 
                                       value="<?= isset($rincian['biaya_pesawat_berangkat']) ? number_format($rincian['biaya_pesawat_berangkat'], 0, ',', '.') : '0' ?>" required>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Biaya Pesawat Kembali</label>
                            <div class="input-group">
                                <span class="input-group-text">Rp</span>
                                <input type="text" class="form-control money" name="pesawat_kembali" 
                                       value="<?= isset($rincian['biaya_pesawat_pulang']) ? number_format($rincian['biaya_pesawat_pulang'], 0, ',', '.') : '0' ?>" required>
                            </div>
                        </div>
                    <?php endif; ?>

                    <?php if ($surat['jenis_surat'] != 'Dalam Daerah Dalam Provinsi'): ?>
                    <div class="mb-3">
                        <label class="form-label">Keterangan</label>
                        <textarea class="form-control" name="keterangan" rows="3"><?= isset($rincian['keterangan']) ? htmlspecialchars($rincian['keterangan']) : '' ?></textarea>
                    </div>
                    <?php endif; ?>

                    <div class="d-flex justify-content-end gap-2">
                    <a href="detail_rincian.php?id=<?= $id_surat ?>" class="btn btn-secondary">
                            <i class="bi bi-arrow-left"></i> Kembali
                        </a>
                        <button type="submit" class="btn btn-primary">Simpan Rincian</button>
                    </div>
                </form>
            </div>
        </div>

        <?php else: ?>
        <div class="alert alert-danger">
            Data surat tugas tidak ditemukan.
        </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
    // Format input uang
    $('.money').on('input', function() {
        let value = $(this).val().replace(/[^\d]/g, '');
        $(this).val(new Intl.NumberFormat('id-ID').format(value));
    });

    // Fungsi format rupiah
    function formatRupiah(angka) {
        return 'Rp ' + new Intl.NumberFormat('id-ID').format(angka);
    }

    // Hitung total uang harian (selalu 2 hari untuk semua jenis surat)
    function hitungTotalUangHarian() {
        let perHari = parseInt($('#uang_harian_per_hari').val().replace(/[^\d]/g, '') || 0);
        let total = perHari * 2; // Selalu 2 hari (hari pertama dan terakhir)
        
        // Tampilkan perhitungan
        $('#perhitungan_uang_harian').show();
        $('#rincian_uang_harian').html(formatRupiah(perHari) + ' × 2 hari');
        $('#total_uang_harian').html(formatRupiah(total));

        hitungTotal();
    }

    // Hitung total uang diklat (sisa hari)
    function hitungTotalUangDiklat() {
        let perHari = parseInt($('#uang_diklat_per_hari').val().replace(/[^\d]/g, '') || 0);
        let lamaPerjalanan = <?= $surat['lama_perjalanan'] ?>;
        let sisaHari = Math.max(0, lamaPerjalanan - 2); // Kurangi 2 hari uang harian
        let total = perHari * sisaHari;
        
        // Tampilkan perhitungan
        $('#perhitungan_uang_diklat').show();
        $('#rincian_uang_diklat').html(formatRupiah(perHari) + ' × ' + sisaHari + ' hari');
        $('#total_uang_diklat').html(formatRupiah(total));

        hitungTotal();
    }

    // Hitung biaya jarak (untuk Dalam Daerah Dalam Provinsi)
    function hitungBiayaJarak() {
        const jarak = document.getElementById('jarak').value || 0;
        const biayaPerKm =5000; // Biaya per kilometer
        const totalBiaya = jarak * biayaPerKm;
        
        // Format ke rupiah dan tampilkan
        document.getElementById('biaya_jarak').value = new Intl.NumberFormat('id-ID').format(totalBiaya);
        
        // Tampilkan perhitungan
        $('#perhitungan_biaya_jarak').show();
        $('#rincian_biaya_jarak').html(jarak + ' KM × ' + formatRupiah(biayaPerKm) + ' per KM');
        $('#total_biaya_jarak').html(formatRupiah(totalBiaya));
        
        hitungTotal();
    }

    // Fungsi untuk menghitung total representatif
    function hitungTotalRepresentatif() {
        let perHari = parseInt($('#representatif_per_hari').val().replace(/[^\d]/g, '') || 0);
        let lamaPerjalanan = <?= $surat['lama_perjalanan'] ?>;
        let total = perHari * lamaPerjalanan;
        
        // Tampilkan perhitungan
        $('#perhitungan_representatif').show();
        $('#rincian_representatif').html(formatRupiah(perHari) + ' × ' + lamaPerjalanan + ' hari');
        $('#total_representatif').html(formatRupiah(total));
        
        // Set nilai total ke hidden input
        $('#representatif_total').val(total);
    }

    // Panggil fungsi saat halaman dimuat
    $(document).ready(function() {
        hitungTotalUangHarian();
        hitungTotalUangDiklat();
        <?php if ($show_representatif): ?>
        hitungTotalRepresentatif();
        <?php endif; ?>
    });
    </script>
</body>
</html> 