<?php
session_start();
require_once '../config/config.php';
require_once '../config/database.php';

// Fungsi untuk memformat ukuran file
function formatFileSize($bytes) {
    if ($bytes === 0) return '0 Bytes';
    $k = 1024;
    $sizes = ['Bytes', 'KB', 'MB', 'GB'];
    $i = floor(log($bytes) / log($k));
    return round($bytes / pow($k, $i), 2) . ' ' . $sizes[$i];
}

// Cek login
if (!isset($_SESSION['user_id'])) {
    header("Location: ../login.php");
    exit();
}

// Ambil info_id dari parameter
$info_id = isset($_GET['info_id']) ? (int)$_GET['info_id'] : 0;

// Ambil detail informasi
$query_info = "SELECT i.*, u.username, u.nama_lengkap, u.bidang, u.level 
               FROM informasi i 
               LEFT JOIN users u ON i.created_by = u.id 
               WHERE i.id_info = ?";
$stmt = mysqli_prepare($conn, $query_info);
mysqli_stmt_bind_param($stmt, "i", $info_id);
mysqli_stmt_execute($stmt);
$result_info = mysqli_stmt_get_result($stmt);
$info = mysqli_fetch_assoc($result_info);

// Jika informasi tidak ditemukan
if (!$info) {
    header("Location: ../dashboard.php");
    exit();
}

// Cek akses ke chat room
$has_access = false;
if ($_SESSION['level'] === 'admin') {
    $has_access = true;
} elseif ($_SESSION['level'] === 'admin_sppd') {
    // Admin SPPD bisa mengakses chat jika:
    // 1. Info dari admin
    // 2. Info dari bidangnya
    // 3. Info yang dia buat
    // 4. Info dari bandahara
    $has_access = 
        $info['level'] === 'admin' ||
        $info['bidang'] === $_SESSION['bidang'] ||
        $_SESSION['user_id'] === $info['created_by'] ||
        $info['level'] === 'bandahara';
} elseif ($_SESSION['level'] === 'pimpinan') {
    // Pimpinan bisa mengakses chat jika:
    // 1. Info dari admin
    // 2. Info dari bidangnya
    // 3. Info yang dia buat sendiri
    // 4. Info dari bandahara
    $has_access = 
        $info['level'] === 'admin' ||
        $info['bidang'] === $_SESSION['bidang'] ||
        $_SESSION['user_id'] === $info['created_by'] ||
        $info['level'] === 'bandahara';
} else {
    // Pengguna lain bisa mengakses chat jika:
    // 1. Informasi dari bidang mereka
    // 2. Mereka adalah pembuat informasi
    // 3. Informasi dari bandahara
    $has_access = 
        $_SESSION['bidang'] === $info['bidang'] ||
        $_SESSION['user_id'] === $info['created_by'] ||
        $info['level'] === 'bandahara';
}

if (!$has_access) {
    header("Location: ../dashboard.php");
    exit();
}

// Ambil pesan chat untuk informasi ini
$query_chat = "SELECT DISTINCT c.id, c.message, c.message_type, c.created_at, c.user_id,
               u.username, u.nama_lengkap, u.bidang, u.foto,
               cf.id as file_id, cf.file_name, cf.original_name, cf.file_type, cf.file_size 
               FROM chat_messages c
               LEFT JOIN users u ON c.user_id = u.id
               LEFT JOIN chat_files cf ON c.id = cf.message_id
               WHERE c.info_id = ? 
               GROUP BY c.id
               ORDER BY c.created_at ASC";
$stmt = mysqli_prepare($conn, $query_chat);
mysqli_stmt_bind_param($stmt, "i", $info_id);
mysqli_stmt_execute($stmt);
$result_chat = mysqli_stmt_get_result($stmt);
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Room Chat - <?= htmlspecialchars($info['judul']) ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyNCAyNCI+PHBhdGggZmlsbD0iIzRlNzNkZiIgZD0iTTIwLDhINFY2aDIwdjJIMjB6IE0xNSwxMEg0djJoMTFWMTB6IE00LDE2aDd2Mkg0VjE2eiBNMjEuNywxMy4zTDIwLjMsMTJsMS40LTEuNEMyMi4xLDEwLjIsMjIuMSw5LjgsMjEuNyw5LjRsLTEuMS0xLjFDMjAuMiw3LjksMTkuOCw3LjksMTkuNCw4LjNMMTgsMTBsLTEuNC0xLjRjLTAuNC0wLjQtMS0wLjQtMS40LDBsLTcuNSw3LjVjLTAuMiwwLjItMC4zLDAuNC0wLjMsMC43djIuOWMwLDAuNSwwLjQsMC45LDAuOSwwLjhoMi45YzAuMywwLDAuNS0wLjEsMC43LTAuM2w3LjUtNy41QzIyLjEsMTQuMywyMi4xLDEzLjcsMjEuNywxMy4zeiIvPjwvc3ZnPg==">
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.1/font/bootstrap-icons.css">
    
    <!-- Emoji Picker Element -->
    <script type="module" src="https://cdn.jsdelivr.net/npm/emoji-picker-element@1/index.js"></script>
    
    <!-- Audio element for notification -->
    <audio id="notificationSound" preload="auto">
        <source src="../assets/sounds/mixkit-bell-notification-933.mp3" type="audio/mpeg">
        <source src="../assets/sounds/mixkit-bell-notification-933.ogg" type="audio/ogg">
        Your browser does not support the audio element.
    </audio>
    
    <!-- Custom CSS -->
    <style>
        :root {
            --primary-gradient: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
            --bg-gradient: linear-gradient(135deg, rgb(245, 255, 254) 0%, rgb(153, 224, 238) 100%);
        }

        body {
            background: var(--bg-gradient) !important;
            min-height: 100vh;
            font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
        }

        .chat-container {
            max-width: 1200px;
            margin: 2rem auto;
            background: white;
            border-radius: 20px;
            box-shadow: 0 8px 32px rgba(31, 38, 135, 0.1);
            overflow: hidden;
        }

        .chat-header {
            background: var(--primary-gradient);
            color: white;
            padding: 1.5rem;
            border-radius: 20px 20px 0 0;
        }

        .chat-header h4 {
            margin: 0;
            font-weight: 600;
        }

        .info-section {
            background: rgba(255, 255, 255, 0.9);
            border-bottom: 1px solid rgba(0,0,0,0.1);
            padding: 1.5rem;
        }

        .info-badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-size: 0.875rem;
            font-weight: 500;
            background: var(--primary-gradient);
            color: white;
            margin-bottom: 1rem;
            display: inline-block;
        }

        .chat-messages {
            height: calc(100vh - 400px);
            overflow-y: auto;
            padding: 1.5rem;
            background: rgba(245, 255, 254, 0.5);
            scroll-behavior: smooth;
            overscroll-behavior-y: contain;
            -webkit-overflow-scrolling: touch;
        }

        .message {
            margin-bottom: 1.5rem;
            max-width: 80%;
            display: flex;
            align-items: flex-start;
            gap: 1rem;
        }

        .message.sent {
            margin-left: auto;
            flex-direction: row-reverse;
        }

        .message-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            overflow: hidden;
            flex-shrink: 0;
            background: var(--primary-gradient);
            position: relative;
        }

        .message-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            position: absolute;
            top: 0;
            left: 0;
        }

        .message-avatar .default-avatar {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.2rem;
            font-weight: 500;
            text-transform: uppercase;
        }

        .message-avatar img.error {
            display: none;
        }

        /* Hover effect for avatar */
        .message-avatar:hover {
            transform: scale(1.1);
            transition: transform 0.2s ease;
        }

        .message-wrapper {
            flex: 1;
        }

        .message-content {
            background: white;
            padding: 1rem;
            border-radius: 15px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            position: relative;
            margin-bottom: 0.5rem;
        }

        .message.sent .message-content {
            background: var(--primary-gradient);
            color: white;
        }

        .message-info {
            font-size: 0.875rem;
            color: #666;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .message.sent .message-info {
            justify-content: flex-end;
            color: #444;
        }

        .chat-input {
            padding: 1.5rem;
            background: white;
            border-top: 1px solid rgba(0,0,0,0.1);
        }

        .chat-input form {
            display: flex;
            gap: 1rem;
            align-items: flex-start;
        }

        .chat-input textarea {
            flex: 1;
            border: 2px solid #e2e8f0;
            border-radius: 15px;
            padding: 0.75rem 2.5rem 0.75rem 0.75rem;
            resize: none;
            height: 60px;
        }

        .chat-input textarea:focus {
            border-color: #0ea5e9;
            box-shadow: 0 0 0 2px rgba(14, 165, 233, 0.15);
            outline: none;
        }

        .btn-file {
            background: transparent;
            border: none;
            color: #666;
            padding: 0.75rem;
            font-size: 1.2rem;
            cursor: pointer;
            transition: color 0.3s ease;
            position: absolute;
            right: 40px;
            top: 50%;
            transform: translateY(-50%);
            z-index: 2;
        }

        .btn-file:hover {
            color: #0ea5e9;
        }

        .file-preview {
            max-width: 200px;
            max-height: 200px;
            object-fit: cover;
            border-radius: 8px;
            margin: 10px 0;
            border: 2px solid rgba(255, 255, 255, 0.1);
        }

        .file-message {
            background: #f8fafc;
            padding: 10px;
            border-radius: 8px;
            margin: 5px 0;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        /* Khusus untuk pesan yang dikirim (sent) */
        .message.sent .file-message {
            background: rgba(255, 255, 255, 0.9) !important;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .message.sent .file-message i,
        .message.sent .file-message .file-name,
        .message.sent .file-message .file-size {
            color: #0ea5e9 !important;
        }

        .file-message i {
            font-size: 24px;
            color: #0ea5e9;
        }

        .file-message .file-info {
            flex: 1;
        }

        .file-message .file-name {
            font-weight: 500;
            margin-bottom: 2px;
            color: #1e293b;
        }

        .file-message .file-size {
            font-size: 12px;
            color: #64748b;
        }

        /* Styling untuk tombol download */
        .file-message .btn-download {
            padding: 5px 10px;
            background: var(--primary-gradient);
            border: none;
            border-radius: 5px;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            justify-content: center;
        }

        /* Tombol download untuk pesan yang diterima */
        .message:not(.sent) .file-message .btn-download {
            background: var(--primary-gradient);
            color: white;
        }

        .message:not(.sent) .file-message .btn-download i {
            color: white;
        }

        /* Tombol download untuk pesan yang dikirim */
        .message.sent .file-message .btn-download {
            background: white;
            color: #0ea5e9;
        }

        .message.sent .file-message .btn-download i {
            color: #0ea5e9;
        }

        .file-message .btn-download:hover {
            transform: translateY(-2px);
            box-shadow: 0 2px 8px rgba(14, 165, 233, 0.2);
        }

        #fileInput {
            display: none;
        }

        .emoji-picker {
            position: absolute;
            bottom: calc(100% + 10px);
            right: 0;
            z-index: 1000;
            display: none;
        }

        .emoji-picker.visible {
            display: block;
        }

        .btn-emoji {
            background: transparent;
            border: none;
            color: #666;
            padding: 0.75rem;
            font-size: 1.2rem;
            cursor: pointer;
            transition: color 0.3s ease;
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            z-index: 2;
        }

        .btn-emoji:hover {
            color: #0ea5e9;
        }

        .input-group {
            position: relative;
            flex: 1;
        }

        /* Mobile responsive styles */
        @media (max-width: 768px) {
            .emoji-picker {
                position: fixed;
                bottom: 80px;
                right: 10px;
            }
        }

        .btn-send {
            background: var(--primary-gradient);
            color: white;
            border: none;
            border-radius: 15px;
            padding: 0.75rem 2rem;
            font-weight: 500;
            transition: all 0.3s ease;
            height: 60px;
        }

        .btn-send:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(14, 165, 233, 0.2);
        }

        /* Custom Scrollbar */
        .chat-messages::-webkit-scrollbar {
            width: 6px;
        }

        .chat-messages::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 3px;
        }

        .chat-messages::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
            border-radius: 3px;
        }

        .chat-messages::-webkit-scrollbar-thumb:hover {
            background: linear-gradient(135deg, #0c8bc3 0%, #25b3a1 100%);
        }

        /* Tambahkan padding bottom pada container pesan terakhir */
        .chat-messages .message:last-child {
            margin-bottom: 1rem;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .chat-container {
                margin: 0;
                border-radius: 0;
                height: 100vh;
            }

            .chat-header {
                border-radius: 0;
            }

            .chat-messages {
                height: calc(100vh - 350px);
            }

            .message {
                max-width: 90%;
            }
        }

        /* Responsive adjustments */
        @media (max-width: 576px) {
            .message-avatar {
                width: 32px;
                height: 32px;
            }
            .message-avatar .default-avatar {
                font-size: 1rem;
            }
        }
    </style>
</head>
<body>
    <div class="chat-container">
        <div class="chat-header">
            <div class="d-flex justify-content-between align-items-center">
                <h4>
                    <i class="bi bi-chat-dots-fill me-2"></i>
                    Room Chat
                </h4>
                <a href="../dashboard.php" class="btn btn-back">
                    <i class="bi bi-arrow-left me-2"></i>
                    Kembali
                </a>
            </div>
        </div>

        <div class="info-section">
            <div class="info-badge">
                <i class="bi bi-info-circle me-2"></i>
                Informasi Terkait
            </div>
            <h5><?= htmlspecialchars($info['judul']) ?></h5>
            <p class="mb-2"><?= nl2br(htmlspecialchars($info['isi'])) ?></p>
            <small class="text-muted">
                <i class="bi bi-person me-1"></i>
                <?= htmlspecialchars($info['nama_lengkap'] ?: $info['username']) ?>
                <span class="mx-2">•</span>
                <i class="bi bi-clock me-1"></i>
                <?= date('d/m/Y H:i', strtotime($info['created_at'])) ?>
            </small>
        </div>

        <div class="chat-messages" id="chatMessages" data-last-message-id="<?php
            $last_message = mysqli_query($conn, "SELECT MAX(id) as last_id FROM chat_messages WHERE info_id = $info_id");
            $last_id = mysqli_fetch_assoc($last_message)['last_id'] ?? 0;
            echo $last_id;
        ?>">
            <?php while ($message = mysqli_fetch_assoc($result_chat)): ?>
                <div class="message <?= $message['user_id'] == $_SESSION['user_id'] ? 'sent' : '' ?>" data-message-id="<?= $message['id'] ?>">
                    <div class="message-avatar">
                        <?php 
                        $profile_pic = '';
                        if (!empty($message['foto'])) {
                            $profile_pic = "../assets/img/users/" . htmlspecialchars($message['foto']);
                            if (!file_exists($profile_pic)) {
                                $profile_pic = '';
                            }
                        }
                        
                        if ($profile_pic): 
                        ?>
                            <img src="<?= $profile_pic ?>" 
                                 alt="Profile" 
                                 onerror="this.classList.add('error'); this.parentElement.querySelector('.default-avatar').style.display='flex';">
                            <div class="default-avatar" style="display: none;">
                                <?= strtoupper(substr($message['nama_lengkap'] ?: $message['username'], 0, 1)) ?>
                            </div>
                        <?php else: ?>
                            <div class="default-avatar">
                                <?= strtoupper(substr($message['nama_lengkap'] ?: $message['username'], 0, 1)) ?>
                            </div>
                        <?php endif; ?>
                    </div>
                    <div class="message-wrapper">
                        <div class="message-content">
                            <?php if ($message['message_type'] === 'file' && !empty($message['file_id'])): ?>
                                <?php
                                $fileIcon = 'bi-file-earmark';
                                $filePreview = '';
                                $fileUrl = "../assets/uploads/chat/" . htmlspecialchars($message['file_name']);
                                
                                if (strpos($message['file_type'], 'image/') === 0) {
                                    $fileIcon = 'bi-file-image';
                                    $filePreview = '<img src="' . $fileUrl . '" class="file-preview" alt="' . htmlspecialchars($message['original_name']) . '" loading="lazy">';
                                } elseif (strpos($message['file_type'], 'video/') === 0) {
                                    $fileIcon = 'bi-file-play';
                                } elseif (strpos($message['file_type'], 'pdf') !== false) {
                                    $fileIcon = 'bi-file-pdf';
                                } elseif (strpos($message['file_type'], 'word') !== false || strpos($message['file_type'], 'document') !== false) {
                                    $fileIcon = 'bi-file-word';
                                } elseif (strpos($message['file_type'], 'sheet') !== false || strpos($message['file_type'], 'excel') !== false) {
                                    $fileIcon = 'bi-file-excel';
                                }
                                ?>
                                <?= $filePreview ?>
                                <div class="file-message">
                                    <i class="bi <?= $fileIcon ?>"></i>
                                    <div class="file-info">
                                        <div class="file-name"><?= htmlspecialchars($message['original_name']) ?></div>
                                        <div class="file-size"><?= formatFileSize($message['file_size']) ?></div>
                                    </div>
                                    <a href="download_file.php?id=<?= $message['file_id'] ?>" class="btn-download" target="_blank" download>
                                        <i class="bi bi-download"></i>
                                    </a>
                                </div>
                            <?php else: ?>
                                <?= nl2br(htmlspecialchars($message['message'])) ?>
                            <?php endif; ?>
                        </div>
                        <div class="message-info">
                            <span><?= htmlspecialchars($message['nama_lengkap'] ?: $message['username']) ?></span>
                            <span>•</span>
                            <span><?= date('H:i', strtotime($message['created_at'])) ?></span>
                        </div>
                    </div>
                </div>
            <?php endwhile; ?>
        </div>

        <div class="chat-input">
            <form id="chatForm">
                <input type="hidden" name="info_id" value="<?= $info_id ?>">
                <div class="input-group">
                    <textarea name="message" placeholder="Ketik pesan Anda..." required></textarea>
                    <button type="button" class="btn-file" id="fileButton">
                        <i class="bi bi-paperclip"></i>
                    </button>
                    <button type="button" class="btn-emoji" id="emojiButton">
                        <i class="bi bi-emoji-smile"></i>
                    </button>
                    <div class="emoji-picker" id="emojiPicker">
                        <emoji-picker></emoji-picker>
                    </div>
                    <input type="file" id="fileInput" accept="image/*,video/*,.pdf,.doc,.docx,.xls,.xlsx">
                </div>
                <button type="submit" class="btn btn-send">
                    <i class="bi bi-send-fill"></i>
                </button>
            </form>
        </div>
    </div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
    // Fungsi helper yang digunakan di seluruh script
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    function getFileIcon(fileType) {
        if (fileType.startsWith('image/')) return 'bi-file-image';
        if (fileType.startsWith('video/')) return 'bi-file-play';
        if (fileType.includes('pdf')) return 'bi-file-pdf';
        if (fileType.includes('word')) return 'bi-file-word';
        if (fileType.includes('sheet') || fileType.includes('excel')) return 'bi-file-excel';
        return 'bi-file-earmark';
    }

    function createAvatarHTML(userData) {
        const userInitial = userData.user_name.charAt(0).toUpperCase();
        if (userData.user_photo) {
            const profilePicPath = "../assets/img/users/" + userData.user_photo;
            return `
                <img src="${profilePicPath}" 
                     alt="Profile" 
                     onerror="this.classList.add('error'); this.parentElement.querySelector('.default-avatar').style.display='flex';">
                <div class="default-avatar" style="display: none;">
                    ${userInitial}
                </div>
            `;
        }
        return `
            <div class="default-avatar">
                ${userInitial}
            </div>
        `;
    }

    document.addEventListener('DOMContentLoaded', function() {
        const chatMessages = document.getElementById('chatMessages');
        const chatForm = document.getElementById('chatForm');
        const messageInput = chatForm.querySelector('textarea[name="message"]');
        const sendButton = chatForm.querySelector('.btn-send');
        const notificationSound = document.getElementById('notificationSound');
        const emojiButton = document.getElementById('emojiButton');
        const emojiPicker = document.getElementById('emojiPicker');
        const fileInput = document.getElementById('fileInput');
        const fileButton = document.getElementById('fileButton');
        let isSubmitting = false;
        let isWindowActive = true;
        let displayedMessageIds = new Set();
        
        // Inisialisasi displayedMessageIds dengan pesan yang sudah ada
        document.querySelectorAll('.message').forEach(message => {
            const messageId = message.dataset.messageId;
            if (messageId) {
                displayedMessageIds.add(messageId);
            }
        });

        // Fungsi untuk scroll ke bawah
        function scrollToBottom(smooth = false) {
            chatMessages.scrollTo({
                top: chatMessages.scrollHeight,
                behavior: smooth ? 'smooth' : 'auto'
            });
        }

        // Scroll ke bawah saat pertama kali masuk
        scrollToBottom();

        // Tambahkan event listener untuk scroll otomatis saat ada pesan baru
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                if (mutation.addedNodes.length) {
                    scrollToBottom(true);
                }
            });
        });

        observer.observe(chatMessages, { childList: true });

        // File upload functionality
        fileButton.addEventListener('click', () => {
            fileInput.click();
        });

        fileInput.addEventListener('change', async (e) => {
            const file = e.target.files[0];
            if (!file) return;

            // Validasi ukuran file (50MB)
            const maxSize = 50 * 1024 * 1024;
            if (file.size > maxSize) {
                alert('File terlalu besar (maksimal 50MB)');
                fileInput.value = '';
                return;
            }

            // Validasi tipe file
            const allowedTypes = [
                'image/jpeg', 'image/png', 'image/gif',
                'video/mp4', 'video/quicktime',
                'application/pdf',
                'application/msword',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                'application/vnd.ms-excel',
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            ];

            if (!allowedTypes.includes(file.type)) {
                alert('Tipe file tidak diizinkan. File yang diizinkan: gambar, video, PDF, Word, Excel');
                fileInput.value = '';
                return;
            }

            const formData = new FormData();
            formData.append('file', file);
            formData.append('info_id', chatForm.querySelector('[name="info_id"]').value);

            try {
                const response = await fetch('upload_file.php', {
                    method: 'POST',
                    body: formData
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const result = await response.json();

                if (result.success) {
                    // Buat pesan HTML tanpa menggunakan formatFileSize
                    const messageHtml = `
                        <div class="message sent" data-message-id="${result.message_id}">
                            <div class="message-avatar">
                                ${createAvatarHTML({
                                    user_name: result.user_name,
                                    user_photo: result.user_photo
                                })}
                            </div>
                            <div class="message-wrapper">
                                <div class="message-content">
                                    ${result.file.type.startsWith('image/') ? 
                                        `<img src="../assets/uploads/chat/${result.file.file_name}" class="file-preview" alt="${result.file.original_name}" loading="lazy">` : 
                                        ''
                                    }
                                    <div class="file-message">
                                        <i class="bi ${getFileIcon(result.file.type)}"></i>
                                        <div class="file-info">
                                            <div class="file-name">${result.file.original_name}</div>
                                            <div class="file-size">${(result.file.size / 1024).toFixed(1)} KB</div>
                                        </div>
                                        <a href="download_file.php?id=${result.file.id}" class="btn-download" target="_blank" download>
                                            <i class="bi bi-download"></i>
                                        </a>
                                    </div>
                                </div>
                                <div class="message-info">
                                    <span>${result.user_name}</span>
                                    <span>•</span>
                                    <span>${result.time}</span>
                                </div>
                            </div>
                        </div>
                    `;

                    displayedMessageIds.add(result.message_id.toString());
                    chatMessages.dataset.lastMessageId = result.message_id;
                    chatMessages.insertAdjacentHTML('beforeend', messageHtml);
                    scrollToBottom();
                } else {
                    throw new Error(result.message || 'Failed to upload file');
                }
            } catch (error) {
                console.error('Upload error:', error);
                alert(error.message || 'Gagal mengupload file. Silakan coba lagi.');
            }

            // Reset file input
            fileInput.value = '';
        });

        // Emoji picker functionality
        emojiButton.addEventListener('click', () => {
            emojiPicker.classList.toggle('visible');
        });

        // Close emoji picker when clicking outside
        document.addEventListener('click', (e) => {
            if (!emojiButton.contains(e.target) && !emojiPicker.contains(e.target)) {
                emojiPicker.classList.remove('visible');
            }
        });

        // Handle emoji selection
        document.querySelector('emoji-picker').addEventListener('emoji-click', event => {
            const emoji = event.detail.unicode;
            const cursorPos = messageInput.selectionStart;
            const textBefore = messageInput.value.substring(0, cursorPos);
            const textAfter = messageInput.value.substring(cursorPos);
            
            messageInput.value = textBefore + emoji + textAfter;
            messageInput.focus();
            
            // Set cursor position after emoji
            const newCursorPos = cursorPos + emoji.length;
            messageInput.setSelectionRange(newCursorPos, newCursorPos);
        });

        // Fungsi untuk mendapatkan ID pesan terakhir
        function getLastDisplayedMessageId() {
            // Pertama coba ambil dari atribut data
            let lastId = chatMessages.dataset.lastMessageId || '0';
            
            // Kemudian cek pesan terakhir yang ada di DOM
            const messages = chatMessages.querySelectorAll('.message');
            if (messages.length > 0) {
                const lastMessage = messages[messages.length - 1];
                const messageId = lastMessage.dataset.messageId;
                if (messageId && parseInt(messageId) > parseInt(lastId)) {
                    lastId = messageId;
                    // Update data attribute
                    chatMessages.dataset.lastMessageId = lastId;
                }
            }
            
            return lastId;
        }

        // Fungsi untuk memuat pesan baru
        async function loadNewMessages() {
            const infoId = chatForm.querySelector('[name="info_id"]').value;
            const lastId = getLastDisplayedMessageId();
            
            try {
                const response = await fetch(`get_new_messages.php?info_id=${infoId}&last_id=${lastId}`);
                const data = await response.json();
                
                if (data.messages && data.messages.length > 0) {
                    let newMessageReceived = false;
                    
                    data.messages.forEach(message => {
                        const messageKey = message.id.toString();
                        if (!displayedMessageIds.has(messageKey)) {
                            displayedMessageIds.add(messageKey);
                            const messageHtml = createMessageHTML(message);
                            chatMessages.insertAdjacentHTML('beforeend', messageHtml);
                            newMessageReceived = true;
                        }
                    });
                    
                    if (newMessageReceived) {
                        playNotificationSound();
                        scrollToBottom(true);
                    }
                }
            } catch (error) {
                console.error('Error loading messages:', error);
            }
        }

        // Set interval untuk auto-refresh pesan setiap 2 detik
        const loadMessagesInterval = setInterval(loadNewMessages, 2000);

        // Bersihkan interval saat halaman ditutup
        window.addEventListener('beforeunload', () => {
            clearInterval(loadMessagesInterval);
        });

        // Bersihkan interval saat tab/window tidak aktif
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                clearInterval(loadMessagesInterval);
            } else {
                // Muat pesan yang mungkin terlewat saat tab tidak aktif
                loadNewMessages();
                // Mulai interval baru
                setInterval(loadNewMessages, 2000);
            }
        });

        // Window focus tracking
        window.addEventListener('focus', () => isWindowActive = true);
        window.addEventListener('blur', () => isWindowActive = false);

        // Handle form submission
        chatForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            if (isSubmitting) return;
            
            const message = messageInput.value.trim();
            if (!message) return;
            
            try {
                isSubmitting = true;
                sendButton.disabled = true;
                
                const formData = new FormData();
                formData.append('info_id', <?= $info_id ?>);
                formData.append('message', message);
                
                const response = await fetch('send_message.php', {
                    method: 'POST',
                    body: formData
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                const data = await response.json();
                
                if (data.success) {
                    messageInput.value = '';
                    const messageHtml = createMessageHTML({
                        id: data.message_id,
                        message: data.message,
                        is_sender: true,
                        user_name: data.user_name,
                        user_photo: data.user_photo,
                        time: data.time
                    });
                    
                    displayedMessageIds.add(data.message_id.toString());
                    chatMessages.insertAdjacentHTML('beforeend', messageHtml);
                    scrollToBottom();
                } else {
                    console.error('Failed to send message:', data.message);
                    if (data.error) {
                        console.error('Server error:', data.error);
                    }
                    alert('Gagal mengirim pesan: ' + (data.message || 'Silakan coba lagi.'));
                }
            } catch (error) {
                console.error('Error sending message:', error);
                alert('Terjadi kesalahan saat mengirim pesan. Silakan coba lagi.');
            } finally {
                isSubmitting = false;
                sendButton.disabled = false;
            }
        });

        // Fungsi untuk memainkan suara notifikasi
        function playNotificationSound() {
            if (!isWindowActive && notificationSound) {
                try {
                    notificationSound.currentTime = 0;
                    const playPromise = notificationSound.play();
                    
                    if (playPromise !== undefined) {
                        playPromise.catch(error => {
                            console.log('Error playing notification:', error);
                        });
                    }
                } catch (error) {
                    console.error('Error playing notification sound:', error);
                }
            }
        }

        // Fungsi untuk membuat HTML pesan
        function createMessageHTML(message) {
            let messageContent = '';
            
            if (message.message_type === 'file' && message.file) {
                const fileIcon = getFileIcon(message.file.type);
                let filePreview = '';
                const fileUrl = `../assets/uploads/chat/${message.file.file_name}`;
                
                if (message.file.type.startsWith('image/')) {
                    filePreview = `<img src="${fileUrl}" class="file-preview" alt="${message.file.original_name}" loading="lazy">`;
                }
                
                messageContent = `
                    ${filePreview}
                    <div class="file-message">
                        <i class="bi ${fileIcon}"></i>
                        <div class="file-info">
                            <div class="file-name">${message.file.original_name}</div>
                            <div class="file-size">${formatFileSize(message.file.size)}</div>
                        </div>
                        <a href="download_file.php?id=${message.file.id}" class="btn-download" target="_blank" download>
                            <i class="bi bi-download"></i>
                        </a>
                    </div>
                `;
            } else {
                messageContent = message.message.replace(/\n/g, '<br>');
            }

            return `
                <div class="message ${message.is_sender ? 'sent' : ''}" data-message-id="${message.id}">
                    <div class="message-avatar">
                        ${createAvatarHTML({
                            user_name: message.user_name,
                            user_photo: message.user_photo
                        })}
                    </div>
                    <div class="message-wrapper">
                        <div class="message-content">
                            ${messageContent}
                        </div>
                        <div class="message-info">
                            <span>${message.user_name}</span>
                            <span>•</span>
                            <span>${message.time}</span>
                        </div>
                    </div>
                </div>
            `;
        }
    });
    </script>
</body>
</html> 
</html> 