<?php
session_start();
require_once '../config/config.php';
require_once '../config/database.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Fungsi untuk logging
function logError($message, $data = null) {
    $log_file = '../logs/upload_error.log';
    $timestamp = date('Y-m-d H:i:s');
    $log_message = "[$timestamp] $message";
    if ($data) {
        $log_message .= "\nData: " . print_r($data, true);
    }
    $log_message .= "\n------------------------\n";
    
    // Buat direktori logs jika belum ada
    if (!file_exists('../logs')) {
        mkdir('../logs', 0777, true);
    }
    
    file_put_contents($log_file, $log_message, FILE_APPEND);
}

// Cek login
if (!isset($_SESSION['user_id'])) {
    logError('Unauthorized access attempt', $_SESSION);
    die(json_encode(['success' => false, 'message' => 'Unauthorized']));
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    logError('Invalid request method', $_SERVER);
    die(json_encode(['success' => false, 'message' => 'Invalid request method']));
}

// Validasi file
if (!isset($_FILES['file'])) {
    logError('No file in request');
    die(json_encode(['success' => false, 'message' => 'No file uploaded']));
}

if ($_FILES['file']['error'] !== UPLOAD_ERR_OK) {
    $error_message = 'Upload error: ';
    switch ($_FILES['file']['error']) {
        case UPLOAD_ERR_INI_SIZE:
            $error_message .= 'File exceeds upload_max_filesize';
            break;
        case UPLOAD_ERR_FORM_SIZE:
            $error_message .= 'File exceeds MAX_FILE_SIZE';
            break;
        case UPLOAD_ERR_PARTIAL:
            $error_message .= 'Partial upload';
            break;
        case UPLOAD_ERR_NO_FILE:
            $error_message .= 'No file uploaded';
            break;
        case UPLOAD_ERR_NO_TMP_DIR:
            $error_message .= 'Missing temporary folder';
            break;
        case UPLOAD_ERR_CANT_WRITE:
            $error_message .= 'Failed to write file to disk';
            break;
        default:
            $error_message .= 'Unknown error';
    }
    logError($error_message, $_FILES['file']);
    die(json_encode(['success' => false, 'message' => $error_message]));
}

$file = $_FILES['file'];
$info_id = $_POST['info_id'] ?? 0;

// Validasi info_id
if (!$info_id) {
    logError('Invalid info ID', $_POST);
    die(json_encode(['success' => false, 'message' => 'Invalid info ID']));
}

// Validasi tipe file yang diizinkan
$allowed_types = [
    'image/jpeg', 'image/png', 'image/gif', 
    'video/mp4', 'video/quicktime',
    'application/pdf', 
    'application/msword', 
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'application/vnd.ms-excel',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
];

$max_size = 50 * 1024 * 1024; // 50MB

// Validasi tipe file menggunakan finfo
$finfo = finfo_open(FILEINFO_MIME_TYPE);
$actual_type = finfo_file($finfo, $file['tmp_name']);
finfo_close($finfo);

if (!in_array($actual_type, $allowed_types)) {
    logError('Invalid file type', [
        'actual_type' => $actual_type,
        'allowed_types' => $allowed_types
    ]);
    die(json_encode([
        'success' => false, 
        'message' => 'File type not allowed. Allowed types: images, videos, PDF, Word, Excel'
    ]));
}

if ($file['size'] > $max_size) {
    logError('File too large', [
        'file_size' => $file['size'],
        'max_size' => $max_size
    ]);
    die(json_encode(['success' => false, 'message' => 'File too large (max 50MB)']));
}

// Buat direktori jika belum ada
$upload_dir = '../assets/uploads/chat';
if (!file_exists($upload_dir)) {
    if (!mkdir($upload_dir, 0777, true)) {
        logError('Failed to create upload directory', [
            'dir' => $upload_dir,
            'error' => error_get_last()
        ]);
        die(json_encode(['success' => false, 'message' => 'Failed to create upload directory']));
    }
}

// Generate nama file unik
$file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
$unique_filename = uniqid('chat_') . '_' . time() . '.' . $file_ext;
$upload_path = $upload_dir . '/' . $unique_filename;

// Pindahkan file
if (!move_uploaded_file($file['tmp_name'], $upload_path)) {
    $upload_error = error_get_last();
    logError('Failed to move uploaded file', [
        'from' => $file['tmp_name'],
        'to' => $upload_path,
        'error' => $upload_error
    ]);
    die(json_encode([
        'success' => false, 
        'message' => 'Failed to save file'
    ]));
}

try {
    // Begin transaction
    mysqli_begin_transaction($conn);

    // Insert chat message with type 'file'
    $query = "INSERT INTO chat_messages (info_id, user_id, message, message_type, created_at) 
              VALUES (?, ?, '', 'file', NOW())";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "ii", 
        $info_id,
        $_SESSION['user_id']
    );

    if (!mysqli_stmt_execute($stmt)) {
        throw new Exception('Gagal menyimpan pesan: ' . mysqli_error($conn));
    }

    $message_id = mysqli_insert_id($conn);

    // Save file information
    $query = "INSERT INTO chat_files (info_id, user_id, message_id, file_name, original_name, file_type, file_size) 
              VALUES (?, ?, ?, ?, ?, ?, ?)";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "iiisssi", 
        $info_id,
        $_SESSION['user_id'],
        $message_id,
        $unique_filename,
        $file['name'],
        $actual_type,
        $file['size']
    );

    if (!mysqli_stmt_execute($stmt)) {
        throw new Exception('Gagal menyimpan informasi file: ' . mysqli_error($conn));
    }

    $file_id = mysqli_insert_id($conn);

    // Get user information for response
    $query = "SELECT username, nama_lengkap, foto FROM users WHERE id = ?";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "i", $_SESSION['user_id']);
    mysqli_stmt_execute($stmt);
    $user_result = mysqli_stmt_get_result($stmt);
    $user_data = mysqli_fetch_assoc($user_result);

    // Commit transaction
    mysqli_commit($conn);

    // Return success response with complete message data
    echo json_encode([
        'success' => true,
        'message' => 'File berhasil diunggah',
        'message_id' => $message_id,
        'message_type' => 'file',
        'user_name' => $user_data['nama_lengkap'] ?: $user_data['username'],
        'user_photo' => $user_data['foto'],
        'time' => date('H:i'),
        'is_sender' => true,
        'file' => [
            'id' => $file_id,
            'file_name' => $unique_filename,
            'original_name' => $file['name'],
            'type' => $actual_type,
            'size' => $file['size']
        ]
    ]);

} catch (Exception $e) {
    // Rollback transaction
    mysqli_rollback($conn);
    
    // Delete uploaded file
    if (file_exists($upload_path)) {
        unlink($upload_path);
    }
    
    logError('Database error', [
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
    
    die(json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]));
} 