<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Brick Game</title>
    <style>
        body {
            margin: 0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(135deg, #1a237e 0%, #283593 50%, #3949ab 100%);
            font-family: 'Poppins', sans-serif;
            color: white;
        }
        
        .game-container {
            text-align: center;
            position: relative;
        }
        
        canvas {
            background: rgba(0, 0, 0, 0.8);
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0,0,0,0.3);
        }
        
        .game-info {
            position: absolute;
            top: 0;
            right: -150px;
            background: rgba(0, 0, 0, 0.7);
            padding: 20px;
            border-radius: 10px;
            text-align: left;
            min-width: 120px;
        }
        
        .controls {
            color: white;
            margin: 20px 0;
            padding: 15px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            text-align: center;
        }
        
        .btn-back {
            display: inline-block;
            padding: 10px 20px;
            background: #4CAF50;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin-top: 20px;
            transition: all 0.3s ease;
        }
        
        .btn-back:hover {
            background: #388E3C;
            transform: translateY(-2px);
        }
        
        .btn-restart {
            display: inline-block;
            padding: 10px 20px;
            background: #FFC107;
            color: white;
            text-decoration: none;
            border: none;
            border-radius: 5px;
            margin: 20px;
            cursor: pointer;
            font-family: 'Poppins', sans-serif;
            transition: all 0.3s ease;
        }
        
        .btn-restart:hover {
            background: #FFA000;
            transform: translateY(-2px);
        }
        
        .game-over {
            display: none;
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0, 0, 0, 0.9);
            color: white;
            padding: 30px;
            border-radius: 15px;
            text-align: center;
            z-index: 100;
        }

        .power-up {
            position: absolute;
            width: 20px;
            height: 20px;
            border-radius: 50%;
            animation: float 2s infinite ease-in-out;
        }

        @keyframes float {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-10px); }
        }
    </style>
</head>
<body>
    <div class="game-container">
        <canvas id="gameCanvas"></canvas>
        <div class="game-info">
            <div>Score: <span id="score">0</span></div>
            <div>Level: <span id="level">1</span></div>
            <div>Lives: <span id="lives">3</span></div>
            <div>Bricks: <span id="bricks">0</span></div>
        </div>
        <div class="controls">
            ← → : Geser paddle<br>
            Spasi : Tembak (jika power-up)<br>
            P : Pause
        </div>
        <div class="game-over" id="gameOver">
            <h2>Game Over!</h2>
            <p>Score: <span id="finalScore">0</span></p>
            <button class="btn-restart" id="restartButton">Main Lagi</button>
        </div>
        <a href="../dashboard.php" class="btn-back">Kembali ke Dashboard</a>
    </div>

    <script>
        const canvas = document.getElementById('gameCanvas');
        const ctx = canvas.getContext('2d');
        const scoreElement = document.getElementById('score');
        const levelElement = document.getElementById('level');
        const livesElement = document.getElementById('lives');
        const bricksElement = document.getElementById('bricks');
        const gameOverElement = document.getElementById('gameOver');
        const finalScoreElement = document.getElementById('finalScore');
        const restartButton = document.getElementById('restartButton');

        // Set canvas size
        canvas.width = 800;
        canvas.height = 600;

        // Game constants
        const PADDLE_HEIGHT = 10;
        const PADDLE_WIDTH = 100;
        const BALL_RADIUS = 8;
        const BRICK_ROW_COUNT = 5;
        const BRICK_COLUMN_COUNT = 8;
        const BRICK_WIDTH = 80;
        const BRICK_HEIGHT = 20;
        const BRICK_PADDING = 10;
        const BRICK_OFFSET_TOP = 60;
        const BRICK_OFFSET_LEFT = 30;
        const PADDLE_SPEED = 12; // Kecepatan paddle
        const POWER_UP_CHANCE = 0.1;
        const POWER_UP_TYPES = {
            WIDE_PADDLE: { color: '#4CAF50', duration: 10000 },
            MULTI_BALL: { color: '#2196F3', duration: 10000 },
            SLOW_BALL: { color: '#FFC107', duration: 10000 },
            EXTRA_LIFE: { color: '#F44336', duration: 0 }
        };

        // Game state
        let score = 0;
        let level = 1;
        let lives = 3;
        let gameOver = false;
        let isPaused = false;
        let bricks = [];
        let powerUps = [];
        let keyState = {}; // Untuk tracking keyboard state
        let activePowerUps = {
            WIDE_PADDLE: false,
            MULTI_BALL: false,
            SLOW_BALL: false
        };
        let balls = [];
        let paddle = {
            x: (canvas.width - PADDLE_WIDTH) / 2,
            y: canvas.height - PADDLE_HEIGHT - 10,
            width: PADDLE_WIDTH,
            height: PADDLE_HEIGHT,
            dx: 8
        };

        // Initialize game
        function initGame() {
            score = 0;
            level = 1;
            lives = 3;
            gameOver = false;
            isPaused = false;
            powerUps = [];
            activePowerUps = {
                WIDE_PADDLE: false,
                MULTI_BALL: false,
                SLOW_BALL: false
            };
            balls = [createBall()];
            createBricks();
            updateScore();
            gameOverElement.style.display = 'none';
        }

        function createBall() {
            return {
                x: canvas.width / 2,
                y: canvas.height - 30,
                dx: 7, // Kecepatan horizontal lebih cepat
                dy: -7, // Kecepatan vertical lebih cepat
                radius: BALL_RADIUS
            };
        }

        function createBricks() {
            bricks = [];
            for (let c = 0; c < BRICK_COLUMN_COUNT; c++) {
                bricks[c] = [];
                for (let r = 0; r < BRICK_ROW_COUNT; r++) {
                    bricks[c][r] = {
                        x: c * (BRICK_WIDTH + BRICK_PADDING) + BRICK_OFFSET_LEFT,
                        y: r * (BRICK_HEIGHT + BRICK_PADDING) + BRICK_OFFSET_TOP,
                        width: BRICK_WIDTH,
                        height: BRICK_HEIGHT,
                        status: 1
                    };
                }
            }
            updateBrickCount();
        }

        function createPowerUp(x, y) {
            if (Math.random() < POWER_UP_CHANCE) {
                const types = Object.keys(POWER_UP_TYPES);
                const type = types[Math.floor(Math.random() * types.length)];
                powerUps.push({
                    x: x,
                    y: y,
                    type: type,
                    width: 20,
                    height: 20
                });
            }
        }

        function drawBall(ball) {
            ctx.beginPath();
            ctx.arc(ball.x, ball.y, ball.radius, 0, Math.PI * 2);
            ctx.fillStyle = '#FFF';
            ctx.fill();
            ctx.closePath();
        }

        function drawPaddle() {
            ctx.beginPath();
            ctx.rect(paddle.x, paddle.y, paddle.width, paddle.height);
            ctx.fillStyle = '#FFF';
            ctx.fill();
            ctx.closePath();
        }

        function drawBricks() {
            for (let c = 0; c < BRICK_COLUMN_COUNT; c++) {
                for (let r = 0; r < BRICK_ROW_COUNT; r++) {
                    if (bricks[c][r].status === 1) {
                        const brickX = c * (BRICK_WIDTH + BRICK_PADDING) + BRICK_OFFSET_LEFT;
                        const brickY = r * (BRICK_HEIGHT + BRICK_PADDING) + BRICK_OFFSET_TOP;
                        ctx.beginPath();
                        ctx.rect(brickX, brickY, BRICK_WIDTH, BRICK_HEIGHT);
                        ctx.fillStyle = `hsl(${(c * r * 20) % 360}, 70%, 50%)`;
                        ctx.fill();
                        ctx.closePath();
                    }
                }
            }
        }

        function drawPowerUps() {
            powerUps.forEach(powerUp => {
                ctx.beginPath();
                ctx.arc(powerUp.x, powerUp.y, powerUp.width / 2, 0, Math.PI * 2);
                ctx.fillStyle = POWER_UP_TYPES[powerUp.type].color;
                ctx.fill();
                ctx.closePath();
            });
        }

        function collisionDetection(ball) {
            for (let c = 0; c < BRICK_COLUMN_COUNT; c++) {
                for (let r = 0; r < BRICK_ROW_COUNT; r++) {
                    const b = bricks[c][r];
                    if (b.status === 1) {
                        if (ball.x > b.x && ball.x < b.x + BRICK_WIDTH && ball.y > b.y && ball.y < b.y + BRICK_HEIGHT) {
                            ball.dy = -ball.dy;
                            b.status = 0;
                            score += 10 * level;
                            createPowerUp(b.x + BRICK_WIDTH / 2, b.y + BRICK_HEIGHT / 2);
                            updateScore();
                            updateBrickCount();
                            
                            if (score % 100 === 0) {
                                level++;
                                paddle.width = Math.max(PADDLE_WIDTH / 2, PADDLE_WIDTH - (level * 10));
                                updateScore();
                            }
                            
                            if (score === BRICK_ROW_COUNT * BRICK_COLUMN_COUNT * 10 * level) {
                                level++;
                                createBricks();
                                balls.forEach(ball => {
                                    ball.dx *= 1.2;
                                    ball.dy *= 1.2;
                                });
                            }
                        }
                    }
                }
            }
        }

        function updatePowerUps() {
            powerUps = powerUps.filter(powerUp => {
                powerUp.y += 2;
                
                // Check collision with paddle
                if (powerUp.y + powerUp.height > paddle.y && 
                    powerUp.x > paddle.x && 
                    powerUp.x < paddle.x + paddle.width) {
                    activatePowerUp(powerUp.type);
                    return false;
                }
                
                // Remove if fallen off screen
                return powerUp.y < canvas.height;
            });
        }

        function activatePowerUp(type) {
            switch(type) {
                case 'WIDE_PADDLE':
                    paddle.width = PADDLE_WIDTH * 2;
                    activePowerUps.WIDE_PADDLE = true;
                    setTimeout(() => {
                        paddle.width = PADDLE_WIDTH;
                        activePowerUps.WIDE_PADDLE = false;
                    }, POWER_UP_TYPES.WIDE_PADDLE.duration);
                    break;
                case 'MULTI_BALL':
                    const currentBalls = [...balls];
                    currentBalls.forEach(ball => {
                        balls.push({
                            ...ball,
                            dx: -ball.dx
                        });
                    });
                    activePowerUps.MULTI_BALL = true;
                    setTimeout(() => {
                        activePowerUps.MULTI_BALL = false;
                    }, POWER_UP_TYPES.MULTI_BALL.duration);
                    break;
                case 'SLOW_BALL':
                    balls.forEach(ball => {
                        ball.dx *= 0.5;
                        ball.dy *= 0.5;
                    });
                    activePowerUps.SLOW_BALL = true;
                    setTimeout(() => {
                        balls.forEach(ball => {
                            ball.dx *= 2;
                            ball.dy *= 2;
                        });
                        activePowerUps.SLOW_BALL = false;
                    }, POWER_UP_TYPES.SLOW_BALL.duration);
                    break;
                case 'EXTRA_LIFE':
                    lives++;
                    updateScore();
                    break;
            }
        }

        function updateBrickCount() {
            let count = 0;
            for (let c = 0; c < BRICK_COLUMN_COUNT; c++) {
                for (let r = 0; r < BRICK_ROW_COUNT; r++) {
                    if (bricks[c][r].status === 1) count++;
                }
            }
            bricksElement.textContent = count;
        }

        function updateScore() {
            scoreElement.textContent = score;
            levelElement.textContent = level;
            livesElement.textContent = lives;
        }

        function draw() {
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            drawBricks();
            drawPaddle();
            balls.forEach(ball => drawBall(ball));
            drawPowerUps();
            
            if (isPaused) {
                ctx.fillStyle = 'rgba(0, 0, 0, 0.5)';
                ctx.fillRect(0, 0, canvas.width, canvas.height);
                ctx.fillStyle = 'white';
                ctx.font = '30px Poppins';
                ctx.textAlign = 'center';
                ctx.fillText('PAUSED', canvas.width / 2, canvas.height / 2);
                ctx.font = '20px Poppins';
                ctx.fillText('Press P to continue', canvas.width / 2, canvas.height / 2 + 40);
            }
        }

        function update() {
            if (gameOver || isPaused) return;

            // Update paddle position based on keyboard state
            if (keyState['ArrowLeft'] || keyState['Left']) {
                paddle.x = Math.max(0, paddle.x - PADDLE_SPEED);
            }
            if (keyState['ArrowRight'] || keyState['Right']) {
                paddle.x = Math.min(canvas.width - paddle.width, paddle.x + PADDLE_SPEED);
            }

            balls = balls.filter(ball => {
                // Ball movement
                ball.x += ball.dx;
                ball.y += ball.dy;

                // Wall collision
                if (ball.x + ball.dx > canvas.width - ball.radius || ball.x + ball.dx < ball.radius) {
                    ball.dx = -ball.dx;
                }
                if (ball.y + ball.dy < ball.radius) {
                    ball.dy = -ball.dy;
                }

                // Paddle collision
                if (ball.y + ball.dy > canvas.height - ball.radius) {
                    if (ball.x > paddle.x && ball.x < paddle.x + paddle.width) {
                        ball.dy = -ball.dy;
                        // Add angle based on where ball hits paddle
                        const hitPoint = (ball.x - (paddle.x + paddle.width / 2)) / (paddle.width / 2);
                        ball.dx = hitPoint * 8; // Meningkatkan efek sudut
                    } else {
                        // Ball missed paddle
                        if (balls.length === 1) {
                            lives--;
                            updateScore();
                            if (lives === 0) {
                                gameOver = true;
                                gameOverElement.style.display = 'block';
                                finalScoreElement.textContent = score;
                                return false;
                            } else {
                                ball.x = canvas.width / 2;
                                ball.y = canvas.height - 30;
                                ball.dx = 7;
                                ball.dy = -7;
                            }
                        } else {
                            return false;
                        }
                    }
                }

                collisionDetection(ball);
                return true;
            });

            updatePowerUps();
            draw();
            requestAnimationFrame(update);
        }

        // Event listeners
        document.addEventListener('keydown', (e) => {
            keyState[e.key] = true;
            
            if (e.key === 'p' || e.key === 'P') {
                isPaused = !isPaused;
                if (!isPaused) update();
            }
        });

        document.addEventListener('keyup', (e) => {
            keyState[e.key] = false;
        });

        document.addEventListener('mousemove', (e) => {
            const relativeX = e.clientX - canvas.offsetLeft;
            if (relativeX > 0 && relativeX < canvas.width) {
                paddle.x = relativeX - paddle.width / 2;
            }
        });

        restartButton.addEventListener('click', () => {
            initGame();
            update();
        });

        // Start game
        initGame();
        update();
    </script>
</body>
</html> 