<?php
session_start();
require_once '../config/config.php';

// Get user's profile photo
$user_id = $_SESSION['user_id'];
$query = "SELECT foto FROM users WHERE id = ?";
$stmt = mysqli_prepare($conn, $query);
mysqli_stmt_bind_param($stmt, "i", $user_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$user = mysqli_fetch_assoc($result);

// Set default photo path
$profile_photo = "../assets/img/users/default.jpg";

// If user has photo, check and set path
if (!empty($user['foto'])) {
    $user_photo_path = "../assets/img/users/" . $user['foto'];
    if (file_exists($user_photo_path)) {
        $profile_photo = $user_photo_path;
    }
}

// Debug info
echo "<!-- Debug: Photo path = " . htmlspecialchars($profile_photo) . " -->";
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Flappy Bird Game</title>
    <style>
        body {
            margin: 0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: #1a237e;
            font-family: 'Poppins', sans-serif;
        }
        
        .game-container {
            text-align: center;
            background: rgba(255, 255, 255, 0.1);
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.3);
        }
        
        canvas {
            background: #87CEEB;
            border: 3px solid #fff;
            border-radius: 8px;
            box-shadow: 0 0 20px rgba(255, 255, 255, 0.2);
        }
        
        .score-board {
            color: white;
            font-size: 28px;
            margin: 20px 0;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            gap: 30px;
        }

        .difficulty-selector {
            margin-bottom: 20px;
            display: flex;
            justify-content: center;
            gap: 10px;
        }

        .difficulty-btn {
            padding: 8px 16px;
            border: none;
            border-radius: 20px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-family: 'Poppins', sans-serif;
            font-weight: bold;
        }

        .difficulty-btn.easy {
            background: #4CAF50;
            color: white;
        }

        .difficulty-btn.medium {
            background: #FFC107;
            color: black;
        }

        .difficulty-btn.hard {
            background: #f44336;
            color: white;
        }

        .difficulty-btn.active {
            transform: scale(1.1);
            box-shadow: 0 0 10px rgba(255,255,255,0.3);
        }
        
        .controls {
            color: white;
            margin-top: 20px;
            padding: 15px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            font-size: 16px;
            line-height: 1.5;
        }
        
        .btn-back {
            display: inline-block;
            padding: 10px 20px;
            background: #00BCD4;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin-top: 20px;
            transition: all 0.3s ease;
            font-weight: bold;
        }
        
        .btn-back:hover {
            background: #0097A7;
            transform: translateY(-2px);
        }

        .btn-restart {
            display: none;
            padding: 12px 24px;
            background: #4CAF50;
            color: white;
            text-decoration: none;
            border: none;
            border-radius: 5px;
            margin-top: 20px;
            cursor: pointer;
            font-family: 'Poppins', sans-serif;
            font-weight: bold;
            font-size: 16px;
            transition: all 0.3s ease;
        }

        .btn-restart:hover {
            background: #45a049;
            transform: translateY(-2px);
        }

        .game-over {
            display: none;
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0, 0, 0, 0.9);
            color: white;
            padding: 30px;
            border-radius: 10px;
            text-align: center;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.5);
        }

        .game-over h2 {
            font-size: 32px;
            margin: 0 0 20px 0;
            color: #f44336;
        }

        .game-over p {
            font-size: 24px;
            margin: 0 0 20px 0;
        }

        .power-up {
            position: absolute;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 16px;
            animation: pulse 1s infinite;
        }

        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); }
        }

        .power-up.shield {
            background: #2196F3;
        }

        .power-up.slow {
            background: #9C27B0;
        }

        .power-up.points {
            background: #FFD700;
        }

        .status-effects {
            position: absolute;
            top: 10px;
            right: 10px;
            display: flex;
            gap: 10px;
        }

        .effect-icon {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 14px;
        }

        .timer {
            position: absolute;
            bottom: 5px;
            right: 5px;
            color: white;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <div class="game-container">
        <div class="score-board">
            <div>Score: <span id="score">0</span></div>
            <div>High Score: <span id="highScore">0</span></div>
        </div>
        <div class="difficulty-selector">
            <button class="difficulty-btn easy active" onclick="setDifficulty('easy')">Mudah</button>
            <button class="difficulty-btn medium" onclick="setDifficulty('medium')">Sedang</button>
            <button class="difficulty-btn hard" onclick="setDifficulty('hard')">Sulit</button>
        </div>
        <canvas id="gameCanvas" width="320" height="480"></canvas>
        <div class="controls">
            Tampahkan foto anda apabila akun anda masih<br>
            menggunakan foto default<br>
            Tekan SPASI untuk melompat<br>
            Kumpulkan power-up untuk kemampuan spesial!<br>
            🛡️ - Perisai (Kebal tabrakan)<br>
            ⏰ - Perlambat Waktu<br>
            ⭐ - Poin Ganda<br>
            Semangat bagawian kakawanan 💪💪💪💪💪💪💪
        </div>
        <div class="game-over" id="gameOver">
            <h2>Game Over!</h2>
            <p>Score: <span id="finalScore">0</span></p>
            <button class="btn-restart" id="restartButton">Main Lagi</button>
        </div>
        <a href="../dashboard.php" class="btn-back">Kembali ke Dashboard</a>
    </div>

    <script>
        const canvas = document.getElementById('gameCanvas');
        const ctx = canvas.getContext('2d');
        const scoreElement = document.getElementById('score');
        const highScoreElement = document.getElementById('highScore');
        const gameOverElement = document.getElementById('gameOver');
        const finalScoreElement = document.getElementById('finalScore');
        const restartButton = document.getElementById('restartButton');

        // Game variables
        let bird = {
            x: 50,
            y: canvas.height / 2,
            velocity: 0,
            width: 40,
            height: 40,
            color: '#FFD700'
        };

        let pipes = [];
        let powerUps = [];
        let score = 0;
        let highScore = localStorage.getItem('flappyHighScore') || 0;
        let gameActive = false;
        
        let activeEffects = {
            shield: false,
            slow: false,
            doublePoints: false
        };
        
        let effectTimers = {
            shield: 0,
            slow: 0,
            doublePoints: 0
        };

        highScoreElement.textContent = highScore;

        // Load bird image
        const birdImage = new Image();
        birdImage.src = '<?php echo $profile_photo; ?>';
        
        // Debug loading status
        birdImage.onload = function() {
            console.log('Image loaded successfully:', this.src);
        };
        
        birdImage.onerror = function(e) {
            console.error('Error loading image:', this.src);
            // Try loading default image
            this.src = '../assets/img/users/default.jpg';
        };

        // Game settings for different difficulties
        const difficulties = {
            easy: { GRAVITY: 0.4, FLAP_SPEED: -7, PIPE_SPEED: 1.5, PIPE_GAP: 170 },
            medium: { GRAVITY: 0.5, FLAP_SPEED: -8, PIPE_SPEED: 2, PIPE_GAP: 150 },
            hard: { GRAVITY: 0.6, FLAP_SPEED: -9, PIPE_SPEED: 2.5, PIPE_GAP: 130 }
        };

        let currentDifficulty = 'easy';
        let gameSettings = difficulties[currentDifficulty];
        let pipeInterval;

        // Event listeners
        document.addEventListener('keydown', function(e) {
            if (e.code === 'Space') {
                e.preventDefault(); // Prevent spacebar from scrolling
                if (!gameActive) {
                    startGame();
                } else {
                    bird.velocity = gameSettings.FLAP_SPEED;
                }
            }
        });

        canvas.addEventListener('click', function() {
            if (!gameActive) {
                startGame();
            } else {
                bird.velocity = gameSettings.FLAP_SPEED;
            }
        });

        restartButton.addEventListener('click', startGame);

        function setDifficulty(difficulty) {
            currentDifficulty = difficulty;
            gameSettings = difficulties[difficulty];
            
            document.querySelectorAll('.difficulty-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            document.querySelector(`.difficulty-btn.${difficulty}`).classList.add('active');
            
            if (gameActive) {
                startGame();
            }
        }

        function createPipe() {
            const gapPosition = Math.random() * (canvas.height - gameSettings.PIPE_GAP - 100) + 50;
            pipes.push({
                x: canvas.width,
                gapY: gapPosition,
                passed: false
            });
        }

        function drawBird() {
            try {
                if (birdImage.complete) {
                    // Draw circular clip
                    ctx.save();
                    ctx.beginPath();
                    ctx.arc(bird.x, bird.y, bird.width/2, 0, Math.PI * 2);
                    ctx.closePath();
                    ctx.clip();

                    // Draw the image
                    ctx.drawImage(
                        birdImage,
                        bird.x - bird.width/2,
                        bird.y - bird.height/2,
                        bird.width,
                        bird.height
                    );
                    
                    ctx.restore();

                    // Draw shield effect if active
                    if (activeEffects.shield) {
                        ctx.beginPath();
                        ctx.arc(bird.x, bird.y, bird.width/2 + 3, 0, Math.PI * 2);
                        ctx.strokeStyle = '#2196F3';
                        ctx.lineWidth = 3;
                        ctx.stroke();
                    }
                } else {
                    // Fallback to circle if image not loaded
                    ctx.beginPath();
                    ctx.arc(bird.x, bird.y, bird.width/2, 0, Math.PI * 2);
                    ctx.fillStyle = bird.color;
                    ctx.fill();
                    if (activeEffects.shield) {
                        ctx.strokeStyle = '#2196F3';
                        ctx.lineWidth = 3;
                        ctx.stroke();
                    }
                }
            } catch (e) {
                console.error('Error in drawBird:', e);
                // Fallback to simple circle
                ctx.beginPath();
                ctx.arc(bird.x, bird.y, bird.width/2, 0, Math.PI * 2);
                ctx.fillStyle = bird.color;
                ctx.fill();
            }
        }

        function drawPipes() {
            ctx.fillStyle = '#4CAF50';
            pipes.forEach(pipe => {
                // Top pipe
                ctx.fillRect(pipe.x, 0, 50, pipe.gapY);
                // Bottom pipe
                ctx.fillRect(
                    pipe.x,
                    pipe.gapY + gameSettings.PIPE_GAP,
                    50,
                    canvas.height - (pipe.gapY + gameSettings.PIPE_GAP)
                );
            });
        }

        function createPowerUp() {
            if (Math.random() < 0.1 && powerUps.length < 2) {
                const types = ['shield', 'slow', 'points'];
                const type = types[Math.floor(Math.random() * types.length)];
                powerUps.push({
                    x: canvas.width,
                    y: Math.random() * (canvas.height - 100) + 50,
                    type: type
                });
            }
        }

        function drawPowerUps() {
            powerUps.forEach(powerUp => {
                ctx.fillStyle = powerUp.type === 'shield' ? '#2196F3' : 
                               powerUp.type === 'slow' ? '#9C27B0' : '#FFD700';
                ctx.beginPath();
                ctx.arc(powerUp.x, powerUp.y, 15, 0, Math.PI * 2);
                ctx.fill();

                ctx.fillStyle = 'white';
                ctx.font = '16px Arial';
                ctx.textAlign = 'center';
                ctx.textBaseline = 'middle';
                ctx.fillText(
                    powerUp.type === 'shield' ? '🛡️' :
                    powerUp.type === 'slow' ? '⏰' : '⭐',
                    powerUp.x,
                    powerUp.y
                );
            });
        }

        function checkPowerUpCollision() {
            powerUps = powerUps.filter(powerUp => {
                if (Math.abs(bird.x - powerUp.x) < 20 && Math.abs(bird.y - powerUp.y) < 20) {
                    activatePowerUp(powerUp.type);
                    return false;
                }
                return powerUp.x > -30;
            });
        }

        function activatePowerUp(type) {
            activeEffects[type] = true;
            effectTimers[type] = 5000;

            switch(type) {
                case 'shield':
                    bird.color = '#2196F3';
                    break;
                case 'slow':
                    gameSettings.PIPE_SPEED *= 0.5;
                    break;
            }
        }

        function updatePowerUps() {
            powerUps.forEach(powerUp => {
                powerUp.x -= gameSettings.PIPE_SPEED;
            });

            Object.keys(effectTimers).forEach(effect => {
                if (activeEffects[effect]) {
                    effectTimers[effect] -= 16;
                    if (effectTimers[effect] <= 0) {
                        deactivatePowerUp(effect);
                    }
                }
            });
        }

        function deactivatePowerUp(type) {
            activeEffects[type] = false;
            switch(type) {
                case 'shield':
                    bird.color = '#FFD700';
                    break;
                case 'slow':
                    gameSettings.PIPE_SPEED = difficulties[currentDifficulty].PIPE_SPEED;
                    break;
            }
        }

        function drawEffectTimers() {
            let y = 10;
            Object.keys(activeEffects).forEach(effect => {
                if (activeEffects[effect]) {
                    const timeLeft = Math.ceil(effectTimers[effect] / 1000);
                    ctx.fillStyle = 'white';
                    ctx.font = '12px Arial';
                    ctx.textAlign = 'right';
                    ctx.fillText(`${effect}: ${timeLeft}s`, canvas.width - 10, y);
                    y += 20;
                }
            });
        }

        function updateGame() {
            if (!gameActive) return;

            bird.velocity += gameSettings.GRAVITY;
            bird.y += bird.velocity;

            updatePowerUps();
            checkPowerUpCollision();

            pipes.forEach(pipe => {
                pipe.x -= gameSettings.PIPE_SPEED;

                if (!pipe.passed && pipe.x < bird.x) {
                    pipe.passed = true;
                    score += activeEffects.doublePoints ? 2 : 1;
                    scoreElement.textContent = score;
                }

                if (checkCollision(pipe) && !activeEffects.shield) {
                    gameOver();
                }
            });

            if (Math.random() < 0.02) {
                createPowerUp();
            }

            pipes = pipes.filter(pipe => pipe.x > -50);

            if (bird.y > canvas.height || bird.y < 0) {
                if (!activeEffects.shield) {
                    gameOver();
                } else {
                    bird.y = Math.max(0, Math.min(bird.y, canvas.height));
                }
            }
        }

        function checkCollision(pipe) {
            return (
                bird.x + bird.width/2 > pipe.x &&
                bird.x - bird.width/2 < pipe.x + 50 &&
                (bird.y - bird.height/2 < pipe.gapY || 
                 bird.y + bird.height/2 > pipe.gapY + gameSettings.PIPE_GAP)
            );
        }

        function draw() {
            // Clear canvas and draw background
            ctx.fillStyle = '#87CEEB';
            ctx.fillRect(0, 0, canvas.width, canvas.height);

            drawBird();
            drawPipes();
            drawPowerUps();
            drawEffectTimers();

            if (!gameActive && pipes.length === 0) {
                ctx.fillStyle = 'white';
                ctx.font = '20px Poppins';
                ctx.textAlign = 'center';
                ctx.fillText('Tekan SPASI untuk Mulai', canvas.width/2, canvas.height/2);
            }
        }

        function gameOver() {
            gameActive = false;
            if (score > highScore) {
                highScore = score;
                localStorage.setItem('flappyHighScore', highScore);
                highScoreElement.textContent = highScore;
            }
            gameOverElement.style.display = 'block';
            restartButton.style.display = 'inline-block';
            finalScoreElement.textContent = score;
            clearInterval(pipeInterval);
        }

        function startGame() {
            // Reset game state
            bird.y = canvas.height / 2;
            bird.velocity = 0;
            bird.color = '#FFD700';
            pipes = [];
            powerUps = [];
            score = 0;
            scoreElement.textContent = score;
            gameActive = true;
            
            // Hide game over screen
            gameOverElement.style.display = 'none';
            restartButton.style.display = 'none';

            // Reset power-ups
            Object.keys(activeEffects).forEach(effect => {
                activeEffects[effect] = false;
                effectTimers[effect] = 0;
            });

            // Reset game settings
            gameSettings = {...difficulties[currentDifficulty]};

            // Clear existing interval and start new one
            if (pipeInterval) clearInterval(pipeInterval);
            createPipe();
            pipeInterval = setInterval(createPipe, 2000);
        }

        function gameLoop() {
            updateGame();
            draw();
            requestAnimationFrame(gameLoop);
        }

        // Start the game loop
        gameLoop();
    </script>
</body>
</html> 