<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Memory Card Game</title>
    <style>
        body {
            margin: 0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(135deg, #1a237e 0%, #283593 50%, #3949ab 100%);
            font-family: 'Poppins', sans-serif;
        }
        
        .game-container {
            text-align: center;
            max-width: 600px;
            margin: auto;
        }
        
        .score-board {
            color: white;
            font-size: 24px;
            margin: 20px 0;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            gap: 30px;
        }

        .level-display {
            color: white;
            font-size: 28px;
            margin-bottom: 20px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.5);
        }

        .memory-game {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 10px;
            perspective: 1000px;
            margin: 20px auto;
            max-width: 800px;
        }

        @media (min-width: 768px) {
            .memory-game.level-2 {
                grid-template-columns: repeat(5, 1fr);
            }
            
            .memory-game.level-3 {
                grid-template-columns: repeat(6, 1fr);
            }
            
            .memory-game.level-4 {
                grid-template-columns: repeat(6, 1fr);
            }
        }

        .memory-card {
            position: relative;
            width: 90px;
            height: 90px;
            cursor: pointer;
            transform-style: preserve-3d;
            transform: scale(1);
            transition: transform 0.5s;
        }

        .memory-card:active {
            transform: scale(0.95);
            transition: transform 0.2s;
        }

        .memory-card.flip {
            transform: rotateY(180deg);
        }

        .front-face,
        .back-face {
            position: absolute;
            width: 100%;
            height: 100%;
            padding: 10px;
            border-radius: 8px;
            background: #fff;
            backface-visibility: hidden;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 2em;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }

        .front-face {
            transform: rotateY(180deg);
            background: #00BCD4;
            color: white;
        }

        .back-face {
            background: #fff;
            transform: rotateY(0deg);
        }
        
        .controls {
            color: white;
            margin-top: 20px;
            padding: 15px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 8px;
        }
        
        .btn-back {
            display: inline-block;
            padding: 10px 20px;
            background: #00BCD4;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin-top: 20px;
            transition: all 0.3s ease;
        }
        
        .btn-back:hover {
            background: #0097A7;
            transform: translateY(-2px);
        }

        .btn-restart {
            display: inline-block;
            padding: 10px 20px;
            background: #4CAF50;
            color: white;
            text-decoration: none;
            border: none;
            border-radius: 5px;
            margin-top: 20px;
            cursor: pointer;
            font-family: 'Poppins', sans-serif;
            transition: all 0.3s ease;
        }

        .btn-restart:hover {
            background: #45a049;
            transform: translateY(-2px);
        }

        .game-over {
            display: none;
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0, 0, 0, 0.9);
            color: white;
            padding: 30px;
            border-radius: 15px;
            text-align: center;
            z-index: 100;
            min-width: 300px;
        }
        
        .next-level-btn {
            display: inline-block;
            padding: 10px 20px;
            background: #FF9800;
            color: white;
            text-decoration: none;
            border: none;
            border-radius: 5px;
            margin: 10px;
            cursor: pointer;
            font-family: 'Poppins', sans-serif;
            transition: all 0.3s ease;
        }
        
        .next-level-btn:hover {
            background: #F57C00;
            transform: translateY(-2px);
        }
    </style>
</head>
<body>
    <div class="game-container">
        <div class="level-display">Level <span id="currentLevel">1</span></div>
        <div class="score-board">
            <div>Moves: <span id="moves">0</span></div>
            <div>Pairs: <span id="pairs">0</span>/<span id="totalPairs">8</span></div>
        </div>
        <div class="memory-game"></div>
        <div class="controls">
            Klik kartu untuk membaliknya<br>
            Temukan semua pasangan kartu yang sama!<br>
            Semangat bagawian kakawanan 💪💪💪💪💪💪💪
        </div>
        <div class="game-over" id="gameOver">
            <h2>Selamat!</h2>
            <p>Kamu menyelesaikan Level <span id="completedLevel">1</span></p>
            <p>dengan <span id="finalMoves">0</span> langkah!</p>
            <button class="next-level-btn" id="nextLevelButton">Level Selanjutnya</button>
            <button class="btn-restart" id="restartButton">Mulai Dari Level 1</button>
        </div>
        <a href="../dashboard.php" class="btn-back">Kembali ke Dashboard</a>
    </div>

    <script>
        const gameContainer = document.querySelector('.memory-game');
        const movesDisplay = document.getElementById('moves');
        const pairsDisplay = document.getElementById('pairs');
        const totalPairsDisplay = document.getElementById('totalPairs');
        const gameOverElement = document.getElementById('gameOver');
        const finalMovesElement = document.getElementById('finalMoves');
        const currentLevelDisplay = document.getElementById('currentLevel');
        const completedLevelDisplay = document.getElementById('completedLevel');
        const nextLevelButton = document.getElementById('nextLevelButton');
        const restartButton = document.getElementById('restartButton');

        let currentLevel = 1;
        let baseEmojis = ['🎨', '🎭', '🎪', '🎢', '🎠', '🎮', '🎲', '🎯', '🎸', '🎺', '🎻', '🎹', '🎼', '🎧', '🎤', '🎬'];
        let emojis = [];
        let cards = [];
        let moves = 0;
        let pairs = 0;
        let hasFlippedCard = false;
        let lockBoard = false;
        let firstCard, secondCard;

        function getLevelConfig(level) {
            const configs = {
                1: { pairs: 8, columns: 4 },
                2: { pairs: 10, columns: 5 },
                3: { pairs: 12, columns: 6 },
                4: { pairs: 15, columns: 6 }
            };
            return configs[level] || configs[4]; // Default to max level config if level is too high
        }

        function createCard(emoji) {
            const card = document.createElement('div');
            card.classList.add('memory-card');
            card.innerHTML = `
                <div class="front-face">${emoji}</div>
                <div class="back-face">❓</div>
            `;
            card.addEventListener('click', flipCard);
            return card;
        }

        function initializeGame(level = 1) {
            currentLevel = level;
            const config = getLevelConfig(level);
            
            // Update displays
            currentLevelDisplay.textContent = level;
            gameContainer.className = 'memory-game level-' + level;
            
            // Reset variables
            moves = 0;
            pairs = 0;
            movesDisplay.textContent = moves;
            pairsDisplay.textContent = pairs;
            
            // Set emojis for current level
            emojis = baseEmojis.slice(0, config.pairs);
            totalPairsDisplay.textContent = config.pairs;
            
            // Clear existing cards
            gameContainer.innerHTML = '';
            
            // Create and shuffle cards
            cards = [...emojis, ...emojis]
                .sort(() => Math.random() - 0.5)
                .map(emoji => createCard(emoji));
            
            // Add cards to container
            cards.forEach(card => gameContainer.appendChild(card));
            
            gameOverElement.style.display = 'none';
        }

        function flipCard() {
            if (lockBoard) return;
            if (this === firstCard) return;

            this.classList.add('flip');

            if (!hasFlippedCard) {
                hasFlippedCard = true;
                firstCard = this;
                return;
            }

            secondCard = this;
            moves++;
            movesDisplay.textContent = moves;
            checkForMatch();
        }

        function checkForMatch() {
            const isMatch = firstCard.querySelector('.front-face').textContent === 
                           secondCard.querySelector('.front-face').textContent;

            isMatch ? disableCards() : unflipCards();
        }

        function disableCards() {
            firstCard.removeEventListener('click', flipCard);
            secondCard.removeEventListener('click', flipCard);
            pairs++;
            pairsDisplay.textContent = pairs;

            const config = getLevelConfig(currentLevel);
            if (pairs === config.pairs) {
                setTimeout(() => {
                    completedLevelDisplay.textContent = currentLevel;
                    finalMovesElement.textContent = moves;
                    gameOverElement.style.display = 'block';
                    
                    // Hide next level button if at max level
                    nextLevelButton.style.display = currentLevel >= 4 ? 'none' : 'inline-block';
                }, 500);
            }

            resetBoard();
        }

        function unflipCards() {
            lockBoard = true;

            setTimeout(() => {
                firstCard.classList.remove('flip');
                secondCard.classList.remove('flip');
                resetBoard();
            }, 1000);
        }

        function resetBoard() {
            [hasFlippedCard, lockBoard] = [false, false];
            [firstCard, secondCard] = [null, null];
        }

        nextLevelButton.addEventListener('click', () => {
            if (currentLevel < 4) {
                initializeGame(currentLevel + 1);
            }
        });

        restartButton.addEventListener('click', () => initializeGame(1));

        // Start the game at level 1
        initializeGame(1);
    </script>
</body>
</html> 