<!DOCTYPE html>
<?php
session_start();
require_once '../config/config.php';

// Simpan skor ke database
function saveScore($score) {
    global $conn;
    $user_id = $_SESSION['user_id'];
    $username = $_SESSION['username'];
    $query = "INSERT INTO snake_scores (user_id, username, score) VALUES (?, ?, ?)";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "isi", $user_id, $username, $score);
    mysqli_stmt_execute($stmt);
}

// Ambil 3 skor tertinggi (hanya skor tertinggi per user)
$query = "SELECT username, MAX(score) as score, created_at 
          FROM snake_scores 
          GROUP BY username 
          ORDER BY score DESC 
          LIMIT 3";
$result = mysqli_query($conn, $query);

// Array untuk menyimpan emoji medali (menggunakan HTML entities untuk memastikan kompatibilitas)
$medals = ['&#129351;', '&#129352;', '&#129353;'];
?>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Snake Dots Game</title>
    <style>
        body {
            margin: 0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: #1a1a1a;
            font-family: 'Courier New', monospace;
        }
        
        .game-container {
            display: flex;
            gap: 30px;
            align-items: flex-start;
            position: relative;
        }

        .game-area {
            text-align: center;
            position: relative;
        }
        
        .leaderboard {
            background: rgba(155, 166, 87, 0.1);
            padding: 20px;
            border-radius: 10px;
            min-width: 250px;
            color: #9ba657;
            margin-left: 30px;
        }

        .leaderboard h2 {
            margin-top: 0;
            text-align: center;
            color: #9ba657;
            border-bottom: 2px solid #9ba657;
            padding-bottom: 10px;
            font-size: 24px;
        }

        .leaderboard-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }

        .leaderboard-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 0;
            border-bottom: 1px solid rgba(155, 166, 87, 0.2);
        }

        .leaderboard-item:last-child {
            border-bottom: none;
        }

        .player-info {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .medal {
            font-size: 24px;
            min-width: 32px;
        }

        .player-details {
            display: flex;
            flex-direction: column;
        }

        .player-name {
            font-weight: bold;
            font-size: 18px;
            color: #b4bf6c;
        }

        .player-score {
            font-weight: bold;
            font-size: 20px;
            color: #d4e157;
        }

        .empty-leaderboard {
            text-align: center;
            padding: 20px;
            color: rgba(155, 166, 87, 0.7);
            font-style: italic;
        }
        
        canvas {
            background: #9ba657;
            border: 10px solid #464646;
            border-radius: 5px;
            image-rendering: pixelated;
        }
        
        .game-info {
            color: #9ba657;
            margin: 20px 0;
            font-size: 20px;
            text-align: center;
            padding-top: 30px;
        }
        
        .controls {
            color: #9ba657;
            margin: 20px 0;
            padding: 15px;
            text-align: center;
            font-size: 16px;
        }
        
        .btn-back {
            display: inline-block;
            padding: 10px 20px;
            background: #9ba657;
            color: #1a1a1a;
            text-decoration: none;
            border-radius: 5px;
            margin-top: 20px;
            font-weight: bold;
            transition: all 0.3s ease;
        }
        
        .btn-back:hover {
            background: #b4bf6c;
            transform: translateY(-2px);
        }
        
        .game-over {
            display: none;
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(26, 26, 26, 0.9);
            color: #9ba657;
            padding: 30px;
            border-radius: 15px;
            text-align: center;
            z-index: 100;
        }
        
        .score-message {
            color: #d4e157;
            margin: 10px 0;
            font-size: 16px;
            font-style: italic;
        }
        
        .btn-restart {
            display: inline-block;
            padding: 10px 20px;
            background: #9ba657;
            color: #1a1a1a;
            text-decoration: none;
            border: none;
            border-radius: 5px;
            margin: 20px;
            cursor: pointer;
            font-family: 'Courier New', monospace;
            font-weight: bold;
            transition: all 0.3s ease;
        }
        
        .btn-restart:hover {
            background: #b4bf6c;
            transform: translateY(-2px);
        }

        .welcome-screen {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(26, 26, 26, 0.9);
            color: #9ba657;
            padding: 30px;
            border-radius: 15px;
            text-align: center;
            z-index: 100;
            width: 80%;
            max-width: 400px;
        }

        .welcome-screen h2 {
            color: #d4e157;
            margin-bottom: 20px;
        }

        .welcome-screen p {
            margin: 10px 0;
            line-height: 1.5;
        }

        .welcome-screen .instructions {
            text-align: left;
            margin: 20px 0;
            padding: 15px;
            background: rgba(155, 166, 87, 0.1);
            border-radius: 8px;
        }

        .welcome-screen .instructions ul {
            list-style-type: none;
            padding: 0;
        }

        .welcome-screen .instructions li {
            margin: 10px 0;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .welcome-screen .key {
            background: #d4e157;
            color: #1a1a1a;
            padding: 2px 8px;
            border-radius: 4px;
            font-weight: bold;
        }

        .btn-start {
            display: inline-block;
            padding: 12px 24px;
            background: #d4e157;
            color: #1a1a1a;
            text-decoration: none;
            border: none;
            border-radius: 5px;
            margin: 20px;
            cursor: pointer;
            font-family: 'Courier New', monospace;
            font-weight: bold;
            font-size: 16px;
            transition: all 0.3s ease;
        }

        .btn-start:hover {
            background: #b4bf6c;
            transform: translateY(-2px);
        }

        .current-user {
            position: absolute;
            top: 10px;
            left: 10px;
            background: rgba(155, 166, 87, 0.1);
            padding: 10px 15px;
            border-radius: 8px;
            color: #9ba657;
            font-size: 14px;
            display: flex;
            align-items: center;
            gap: 8px;
            z-index: 10;
        }

        .user-icon {
            font-size: 18px;
            color: #d4e157;
        }

        .user-name {
            font-weight: bold;
            color: #d4e157;
            text-transform: capitalize;
        }
    </style>
</head>
<body>
    <div class="game-container">
        <div class="game-area">
            <div class="current-user">
                <span class="user-icon">👤</span>
                <span>Player: <span class="user-name"><?php echo htmlspecialchars($_SESSION['username']); ?></span></span>
            </div>
            <div class="game-info">
                Score: <span id="score">0</span>
            </div>
            <canvas id="gameCanvas"></canvas>
            <div class="welcome-screen" id="welcomeScreen">
                <h2>🐍 Snake Dots Game 🎮</h2>
                <p>Selamat datang di permainan Snake Dots!</p>
                <div class="instructions">
                    <p>Cara Bermain:</p>
                    <ul>
                        <li><span class="key">←</span> Gerak ke kiri</li>
                        <li><span class="key">→</span> Gerak ke kanan</li>
                        <li><span class="key">↑</span> Gerak ke atas</li>
                        <li><span class="key">↓</span> Gerak ke bawah</li>
                        <li><span class="key">P</span> Jeda permainan</li>
                    </ul>
                </div>
                <p>Dapatkan skor setinggi mungkin untuk masuk ke dalam Top 3!</p>
                <button class="btn-start" id="startButton">Mulai Main</button>
            </div>
            <div class="controls">
                ← ↑ → ↓ : Kontrol ular<br>
                P : Pause
            </div>
            <div class="game-over" id="gameOver">
                <h2>Game Over!</h2>
                <p>Score: <span id="finalScore">0</span></p>
                <p class="score-message" id="scoreMessage"></p>
                <button class="btn-restart" id="restartButton">Main Lagi</button>
            </div>
            <a href="../dashboard.php" class="btn-back">Kembali ke Dashboard</a>
        </div>

        <div class="leaderboard">
            <h2>🏆 Top Champions 🏆</h2>
            <div class="leaderboard-list">
                <?php
                if (mysqli_num_rows($result) > 0) {
                    $rank = 0;
                    while ($row = mysqli_fetch_assoc($result)) {
                        echo '<div class="leaderboard-item">
                                <div class="player-info">
                                    <span class="medal" style="font-size: 24px;">' . $medals[$rank] . '</span>
                                    <div class="player-details">
                                        <span class="player-name">' . htmlspecialchars($row['username']) . '</span>
                                    </div>
                                </div>
                                <span class="player-score">' . number_format($row['score']) . '</span>
                              </div>';
                        $rank++;
                    }
                } else {
                    echo '<div class="empty-leaderboard">Belum ada pemain</div>';
                }
                ?>
            </div>
        </div>
    </div>

    <script>
        const canvas = document.getElementById('gameCanvas');
        const ctx = canvas.getContext('2d');
        const scoreElement = document.getElementById('score');
        const gameOverElement = document.getElementById('gameOver');
        const welcomeScreen = document.getElementById('welcomeScreen');
        const startButton = document.getElementById('startButton');
        const finalScoreElement = document.getElementById('finalScore');
        const restartButton = document.getElementById('restartButton');

        // Game constants
        const GRID_SIZE = 15;
        const CELL_SIZE = 20;
        const DOT_SIZE = 4;
        const DOT_SPACING = 10;
        
        // Set canvas size
        canvas.width = GRID_SIZE * CELL_SIZE;
        canvas.height = GRID_SIZE * CELL_SIZE;

        // Game state
        let snake = [];
        let food = {};
        let direction = 'right';
        let nextDirection = 'right';
        let score = 0;
        let gameOver = false;
        let isPaused = false;
        let gameLoop;
        let dotPattern = [];
        let gameStarted = false;

        // Initialize dot pattern
        function initDotPattern() {
            dotPattern = [];
            for(let y = 0; y < canvas.height; y += DOT_SPACING) {
                for(let x = 0; x < canvas.width; x += DOT_SPACING) {
                    dotPattern.push({x, y});
                }
            }
        }

        function initGame() {
            // Initialize snake
            snake = [
                {x: 7, y: 7},
                {x: 6, y: 7},
                {x: 5, y: 7}
            ];
            
            // Initialize food
            createFood();
            
            // Reset game state
            direction = 'right';
            nextDirection = 'right';
            score = 0;
            gameOver = false;
            isPaused = false;
            
            // Update score display
            scoreElement.textContent = score;
            gameOverElement.style.display = 'none';
            welcomeScreen.style.display = 'none';
            
            // Clear previous game loop
            if (gameLoop) clearInterval(gameLoop);
            
            // Start game loop
            gameLoop = setInterval(update, 150);
            gameStarted = true;
        }

        function createFood() {
            food = {
                x: Math.floor(Math.random() * GRID_SIZE),
                y: Math.floor(Math.random() * GRID_SIZE)
            };
            
            // Make sure food doesn't appear on snake
            for (let segment of snake) {
                if (segment.x === food.x && segment.y === food.y) {
                    createFood();
                    break;
                }
            }
        }

        function drawDots() {
            ctx.fillStyle = '#727844';
            dotPattern.forEach(dot => {
                ctx.beginPath();
                ctx.arc(dot.x + DOT_SIZE/2, dot.y + DOT_SIZE/2, DOT_SIZE/2, 0, Math.PI * 2);
                ctx.fill();
            });
        }

        function drawSnake() {
            snake.forEach((segment, index) => {
                ctx.fillStyle = index === 0 ? '#1a1a1a' : '#2d2d2d';
                ctx.fillRect(
                    segment.x * CELL_SIZE,
                    segment.y * CELL_SIZE,
                    CELL_SIZE - 1,
                    CELL_SIZE - 1
                );
            });
        }

        function drawFood() {
            ctx.fillStyle = '#1a1a1a';
            ctx.beginPath();
            ctx.arc(
                food.x * CELL_SIZE + CELL_SIZE/2,
                food.y * CELL_SIZE + CELL_SIZE/2,
                CELL_SIZE/2 - 1,
                0,
                Math.PI * 2
            );
            ctx.fill();
        }

        function draw() {
            // Clear canvas
            ctx.fillStyle = '#9ba657';
            ctx.fillRect(0, 0, canvas.width, canvas.height);
            
            // Draw background dots
            drawDots();
            
            // Draw game elements
            drawSnake();
            drawFood();
            
            if (isPaused) {
                ctx.fillStyle = 'rgba(26, 26, 26, 0.7)';
                ctx.fillRect(0, 0, canvas.width, canvas.height);
                ctx.fillStyle = '#9ba657';
                ctx.font = '20px Courier New';
                ctx.textAlign = 'center';
                ctx.fillText('PAUSED', canvas.width/2, canvas.height/2);
                ctx.font = '16px Courier New';
                ctx.fillText('Press P to continue', canvas.width/2, canvas.height/2 + 30);
            }
        }

        function update() {
            if (gameOver || isPaused) return;

            // Update direction
            direction = nextDirection;

            // Calculate new head position
            const head = {x: snake[0].x, y: snake[0].y};
            switch(direction) {
                case 'up': head.y--; break;
                case 'down': head.y++; break;
                case 'left': head.x--; break;
                case 'right': head.x++; break;
            }

            // Check for collisions
            if (head.x < 0 || head.x >= GRID_SIZE || 
                head.y < 0 || head.y >= GRID_SIZE) {
                endGame();
                return;
            }

            // Check for self collision
            for (let segment of snake) {
                if (head.x === segment.x && head.y === segment.y) {
                    endGame();
                    return;
                }
            }

            // Add new head
            snake.unshift(head);

            // Check for food collision
            if (head.x === food.x && head.y === food.y) {
                score += 10;
                scoreElement.textContent = score;
                createFood();
            } else {
                snake.pop();
            }

            draw();
        }

        function endGame() {
            gameOver = true;
            gameOverElement.style.display = 'block';
            finalScoreElement.textContent = score;
            clearInterval(gameLoop);

            // Simpan skor ke database menggunakan AJAX
            fetch('save_score.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'score=' + score
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('scoreMessage').textContent = data.message;
                    // Reload halaman untuk memperbarui leaderboard
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                }
            });
        }

        // Event listeners
        document.addEventListener('keydown', (e) => {
            if (!gameStarted || gameOver) return;

            switch(e.key) {
                case 'ArrowUp':
                    if (direction !== 'down') nextDirection = 'up';
                    break;
                case 'ArrowDown':
                    if (direction !== 'up') nextDirection = 'down';
                    break;
                case 'ArrowLeft':
                    if (direction !== 'right') nextDirection = 'left';
                    break;
                case 'ArrowRight':
                    if (direction !== 'left') nextDirection = 'right';
                    break;
                case 'p':
                case 'P':
                    isPaused = !isPaused;
                    break;
            }
        });

        startButton.addEventListener('click', initGame);
        restartButton.addEventListener('click', () => {
            gameStarted = false;
            welcomeScreen.style.display = 'block';
            gameOverElement.style.display = 'none';
        });

        // Initialize dot pattern and draw initial screen
        initDotPattern();
        draw();
    </script>
</body>
</html> 