<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tetris Game</title>
    <style>
        body {
            margin: 0;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(135deg, #1a237e 0%, #283593 50%, #3949ab 100%);
            font-family: 'Poppins', sans-serif;
            color: white;
        }
        
        .game-container {
            text-align: center;
            position: relative;
        }
        
        canvas {
            background: rgba(0, 0, 0, 0.8);
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0,0,0,0.3);
        }
        
        .game-info {
            position: absolute;
            top: 0;
            right: -150px;
            background: rgba(0, 0, 0, 0.7);
            padding: 20px;
            border-radius: 10px;
            text-align: left;
            min-width: 120px;
        }
        
        .next-piece {
            margin-top: 10px;
            padding: 10px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 5px;
        }
        
        .controls {
            color: white;
            margin: 20px 0;
            padding: 15px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            text-align: center;
        }
        
        .btn-back {
            display: inline-block;
            padding: 10px 20px;
            background: #4CAF50;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin-top: 20px;
            transition: all 0.3s ease;
        }
        
        .btn-back:hover {
            background: #388E3C;
            transform: translateY(-2px);
        }
        
        .btn-restart {
            display: inline-block;
            padding: 10px 20px;
            background: #FFC107;
            color: white;
            text-decoration: none;
            border: none;
            border-radius: 5px;
            margin: 20px;
            cursor: pointer;
            font-family: 'Poppins', sans-serif;
            transition: all 0.3s ease;
        }
        
        .btn-restart:hover {
            background: #FFA000;
            transform: translateY(-2px);
        }
        
        .game-over {
            display: none;
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0, 0, 0, 0.9);
            color: white;
            padding: 30px;
            border-radius: 15px;
            text-align: center;
            z-index: 100;
        }
    </style>
</head>
<body>
    <div class="game-container">
        <canvas id="gameCanvas"></canvas>
        <div class="game-info">
            <div>Score: <span id="score">0</span></div>
            <div>Level: <span id="level">1</span></div>
            <div>Lines: <span id="lines">0</span></div>
            <div class="next-piece">
                <div>Next:</div>
                <canvas id="nextPiece"></canvas>
            </div>
        </div>
        <div class="controls">
            ← → : Geser kiri/kanan<br>
            ↑ : Putar<br>
            ↓ : Turun cepat<br>
            Spasi : Hard drop
        </div>
        <div class="game-over" id="gameOver">
            <h2>Game Over!</h2>
            <p>Score: <span id="finalScore">0</span></p>
            <button class="btn-restart" id="restartButton">Main Lagi</button>
        </div>
        <a href="../dashboard.php" class="btn-back">Kembali ke Dashboard</a>
    </div>

    <script>
        const canvas = document.getElementById('gameCanvas');
        const nextPieceCanvas = document.getElementById('nextPiece');
        const ctx = canvas.getContext('2d');
        const nextCtx = nextPieceCanvas.getContext('2d');
        const scoreElement = document.getElementById('score');
        const levelElement = document.getElementById('level');
        const linesElement = document.getElementById('lines');
        const gameOverElement = document.getElementById('gameOver');
        const finalScoreElement = document.getElementById('finalScore');
        const restartButton = document.getElementById('restartButton');

        // Set canvas size
        canvas.width = 300;
        canvas.height = 600;
        nextPieceCanvas.width = 100;
        nextPieceCanvas.height = 100;

        // Game constants
        const BLOCK_SIZE = 30;
        const BOARD_WIDTH = 10;
        const BOARD_HEIGHT = 20;
        const COLORS = [
            null,
            '#FF0D72', // I
            '#0DC2FF', // J
            '#0DFF72', // L
            '#F538FF', // O
            '#FF8E0D', // S
            '#FFE138', // T
            '#FF0D72'  // Z
        ];

        // Tetromino shapes
        const PIECES = [
            [
                [0, 0, 0, 0],
                [1, 1, 1, 1],
                [0, 0, 0, 0],
                [0, 0, 0, 0]
            ],
            [
                [2, 0, 0],
                [2, 2, 2],
                [0, 0, 0]
            ],
            [
                [0, 0, 3],
                [3, 3, 3],
                [0, 0, 0]
            ],
            [
                [4, 4],
                [4, 4]
            ],
            [
                [0, 5, 5],
                [5, 5, 0],
                [0, 0, 0]
            ],
            [
                [0, 6, 0],
                [6, 6, 6],
                [0, 0, 0]
            ],
            [
                [7, 7, 0],
                [0, 7, 7],
                [0, 0, 0]
            ]
        ];

        let dropCounter = 0;
        let dropInterval = 1000;
        let lastTime = 0;
        let score = 0;
        let level = 1;
        let lines = 0;
        let gameOver = false;

        const player = {
            pos: {x: 0, y: 0},
            matrix: null,
            score: 0
        };

        const arena = createMatrix(BOARD_WIDTH, BOARD_HEIGHT);
        let nextPiece = createPiece(Math.floor(Math.random() * 7));

        function createMatrix(w, h) {
            const matrix = [];
            while (h--) {
                matrix.push(new Array(w).fill(0));
            }
            return matrix;
        }

        function createPiece(type) {
            return PIECES[type];
        }

        function draw() {
            ctx.fillStyle = '#000';
            ctx.fillRect(0, 0, canvas.width, canvas.height);
            
            drawMatrix(arena, {x: 0, y: 0});
            drawMatrix(player.matrix, player.pos);
            
            // Draw next piece
            nextCtx.fillStyle = '#000';
            nextCtx.fillRect(0, 0, nextPieceCanvas.width, nextPieceCanvas.height);
            drawMatrix(nextPiece, {x: 1, y: 1}, nextCtx, 20);
        }

        function drawMatrix(matrix, offset, context = ctx, size = BLOCK_SIZE) {
            matrix.forEach((row, y) => {
                row.forEach((value, x) => {
                    if (value !== 0) {
                        context.fillStyle = COLORS[value];
                        context.fillRect(
                            (x + offset.x) * size,
                            (y + offset.y) * size,
                            size - 1,
                            size - 1
                        );
                    }
                });
            });
        }

        function merge(arena, player) {
            player.matrix.forEach((row, y) => {
                row.forEach((value, x) => {
                    if (value !== 0) {
                        arena[y + player.pos.y][x + player.pos.x] = value;
                    }
                });
            });
        }

        function rotate(matrix, dir) {
            for (let y = 0; y < matrix.length; ++y) {
                for (let x = 0; x < y; ++x) {
                    [
                        matrix[x][y],
                        matrix[y][x],
                    ] = [
                        matrix[y][x],
                        matrix[x][y],
                    ];
                }
            }

            if (dir > 0) {
                matrix.forEach(row => row.reverse());
            } else {
                matrix.reverse();
            }
        }

        function playerDrop() {
            player.pos.y++;
            if (collide(arena, player)) {
                player.pos.y--;
                merge(arena, player);
                playerReset();
                arenaSweep();
                updateScore();
            }
            dropCounter = 0;
        }

        function playerMove(offset) {
            player.pos.x += offset;
            if (collide(arena, player)) {
                player.pos.x -= offset;
            }
        }

        function playerRotate(dir) {
            const pos = player.pos.x;
            let offset = 1;
            rotate(player.matrix, dir);
            while (collide(arena, player)) {
                player.pos.x += offset;
                offset = -(offset + (offset > 0 ? 1 : -1));
                if (offset > player.matrix[0].length) {
                    rotate(player.matrix, -dir);
                    player.pos.x = pos;
                    return;
                }
            }
        }

        function playerReset() {
            player.matrix = nextPiece;
            nextPiece = createPiece(Math.floor(Math.random() * 7));
            player.pos.y = 0;
            player.pos.x = Math.floor(arena[0].length / 2) - 
                          Math.floor(player.matrix[0].length / 2);
            
            if (collide(arena, player)) {
                gameOver = true;
                gameOverElement.style.display = 'block';
                finalScoreElement.textContent = score;
            }
        }

        function collide(arena, player) {
            const [m, o] = [player.matrix, player.pos];
            for (let y = 0; y < m.length; ++y) {
                for (let x = 0; x < m[y].length; ++x) {
                    if (m[y][x] !== 0 &&
                        (arena[y + o.y] &&
                        arena[y + o.y][x + o.x]) !== 0) {
                        return true;
                    }
                }
            }
            return false;
        }

        function arenaSweep() {
            let rowCount = 1;
            outer: for (let y = arena.length - 1; y > 0; --y) {
                for (let x = 0; x < arena[y].length; ++x) {
                    if (arena[y][x] === 0) {
                        continue outer;
                    }
                }

                const row = arena.splice(y, 1)[0].fill(0);
                arena.unshift(row);
                ++y;

                lines += rowCount;
                score += rowCount * 100 * level;
                rowCount *= 2;
                
                // Level up every 10 lines
                if (lines % 10 === 0) {
                    level++;
                    dropInterval = Math.max(100, 1000 - (level * 50));
                }
            }
        }

        function updateScore() {
            scoreElement.textContent = score;
            levelElement.textContent = level;
            linesElement.textContent = lines;
        }

        function update(time = 0) {
            if (gameOver) return;

            const deltaTime = time - lastTime;
            lastTime = time;
            dropCounter += deltaTime;

            if (dropCounter > dropInterval) {
                playerDrop();
            }

            draw();
            requestAnimationFrame(update);
        }

        function hardDrop() {
            while (!collide(arena, player)) {
                player.pos.y++;
            }
            player.pos.y--;
            merge(arena, player);
            playerReset();
            arenaSweep();
            updateScore();
        }

        function initGame() {
            // Reset game state
            arena.forEach(row => row.fill(0));
            score = 0;
            level = 1;
            lines = 0;
            dropInterval = 1000;
            gameOver = false;
            gameOverElement.style.display = 'none';
            updateScore();
            
            // Reset player
            player.pos.x = Math.floor(arena[0].length / 2) - 1;
            player.pos.y = 0;
            player.matrix = createPiece(Math.floor(Math.random() * 7));
            nextPiece = createPiece(Math.floor(Math.random() * 7));
            
            // Start game loop
            update();
        }

        document.addEventListener('keydown', event => {
            if (gameOver) return;

            switch(event.keyCode) {
                case 37: // Left arrow
                    playerMove(-1);
                    break;
                case 39: // Right arrow
                    playerMove(1);
                    break;
                case 40: // Down arrow
                    playerDrop();
                    break;
                case 38: // Up arrow
                    playerRotate(1);
                    break;
                case 32: // Space
                    hardDrop();
                    break;
            }
        });

        restartButton.addEventListener('click', () => {
            initGame();
        });

        // Start game
        initGame();
    </script>
</body>
</html> 