<?php
session_start();
require_once $_SERVER['DOCUMENT_ROOT'] . '/sppd/config/config.php';
require_once '../vendor/autoload.php';

// Pastikan tidak ada output sebelum header
ob_start();

try {
    $conn = check_connection($conn);
    
    // Get rekap ID from URL
    $id_rekap = isset($_GET['id']) ? $_GET['id'] : null;
    
    if (!$id_rekap) {
        throw new Exception("ID Rekap tidak ditemukan");
    }
    
    // Query untuk mendapatkan detail rekap
    $query_rekap = "SELECT r.*, GROUP_CONCAT(dr.id_surat) as selected_surat_ids
                    FROM rekap_sppd r
                    LEFT JOIN detail_rekap_sppd dr ON r.id_rekap = dr.id_rekap
                    WHERE r.id_rekap = ?
                    GROUP BY r.id_rekap";
    
    $stmt = mysqli_prepare($conn, $query_rekap);
    mysqli_stmt_bind_param($stmt, "i", $id_rekap);
    mysqli_stmt_execute($stmt);
    $result_rekap = mysqli_stmt_get_result($stmt);
    $rekap = mysqli_fetch_assoc($result_rekap);
    
    if (!$rekap) {
        throw new Exception("Rekap tidak ditemukan");
    }

    $selected_surat_ids = $rekap['selected_surat_ids'] ? explode(',', $rekap['selected_surat_ids']) : [];

    // Get all available surat tugas
    $query_surat = "SELECT st.id_surat, st.no_surat, st.maksud, st.no_sub_kegiatan, st.total_biaya
                    FROM surat_tugas st
                    LEFT JOIN detail_rekap_sppd dr ON st.id_surat = dr.id_surat
                    WHERE dr.id_surat IS NULL OR dr.id_rekap = ?
                    ORDER BY st.tanggal_surat DESC";
                    
    $stmt = mysqli_prepare($conn, $query_surat);
    mysqli_stmt_bind_param($stmt, "i", $id_rekap);
    mysqli_stmt_execute($stmt);
    $result_surat = mysqli_stmt_get_result($stmt);

    if (!$result_surat) {
        throw new Exception("Error mengambil data surat tugas");
    }

} catch (Exception $e) {
    $_SESSION['alert'] = [
        'type' => 'danger',
        'message' => $e->getMessage()
    ];
    header('Location: index.php');
    exit();
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    try {
        $no_rekap = $_POST['no_rekap'];
        $tanggal_rekap = $_POST['tanggal_rekap'];
        $surat_ids = isset($_POST['surat_ids']) ? $_POST['surat_ids'] : [];
        
        // Start transaction
        mysqli_begin_transaction($conn);
        
        // Update rekap_sppd
        $query_update = "UPDATE rekap_sppd SET no_rekap = ?, tanggal_rekap = ? WHERE id_rekap = ?";
        $stmt = mysqli_prepare($conn, $query_update);
        mysqli_stmt_bind_param($stmt, "ssi", $no_rekap, $tanggal_rekap, $id_rekap);
        
        if (!mysqli_stmt_execute($stmt)) {
            throw new Exception("Error updating rekap SPPD");
        }
        
        // Delete existing detail_rekap_sppd
        $query_delete = "DELETE FROM detail_rekap_sppd WHERE id_rekap = ?";
        $stmt = mysqli_prepare($conn, $query_delete);
        mysqli_stmt_bind_param($stmt, "i", $id_rekap);
        
        if (!mysqli_stmt_execute($stmt)) {
            throw new Exception("Error deleting existing details");
        }
        
        // Insert new detail_rekap_sppd
        $query_insert = "INSERT INTO detail_rekap_sppd (id_rekap, id_surat) VALUES (?, ?)";
        $stmt = mysqli_prepare($conn, $query_insert);
        
        foreach ($surat_ids as $surat_id) {
            mysqli_stmt_bind_param($stmt, "ii", $id_rekap, $surat_id);
            if (!mysqli_stmt_execute($stmt)) {
                throw new Exception("Error inserting detail rekap");
            }
        }
        
        // Commit transaction
        mysqli_commit($conn);
        
        $_SESSION['alert'] = [
            'type' => 'success',
            'message' => 'Rekap SPPD berhasil diupdate'
        ];
        
        header("Location: detail.php?id=$id_rekap");
        exit();
        
    } catch (Exception $e) {
        mysqli_rollback($conn);
        $_SESSION['alert'] = [
            'type' => 'danger',
            'message' => 'Error: ' . $e->getMessage()
        ];
    }
}


?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Rekap SPPD - SPPD System</title>
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyNCAyNCI+PHBhdGggZmlsbD0iIzRlNzNkZiIgZD0iTTIwLDhINFY2aDIwdjJIMjB6IE0xNSwxMEg0djJoMTFWMTB6IE00LDE2aDd2Mkg0VjE2eiBNMjEuNywxMy4zTDIwLjMsMTJsMS40LTEuNEMyMi4xLDEwLjIsMjIuMSw5LjgsMjEuNyw5LjRsLTEuMS0xLjFDMjAuMiw3LjksMTkuOCw3LjksMTkuNCw4LjNMMTgsMTBsLTEuNC0xLjRjLTAuNC0wLjQtMS0wLjQtMS40LDBsLTcuNSw3LjVjLTAuMiwwLjItMC4zLDAuNC0wLjMsMC43djIuOWMwLDAuNSwwLjQsMC45LDAuOSwwLjloMi45YzAuMywwLDAuNS0wLjEsMC43LTAuM2w3LjUtNy41QzIyLjEsMTQuMywyMi4xLDEzLjcsMjEuNywxMy4zeiIvPjwvc3ZnPg==">
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <!-- DataTables CSS -->
    <link href="https://cdn.datatables.net/1.10.24/css/dataTables.bootstrap5.min.css" rel="stylesheet">
</head>
<body>
    <div class="container-fluid mt-3">
        <div class="header-wrapper">
            <h1>Edit Rekap SPPD</h1>
            <p>Edit informasi rekap SPPD</p>
            <nav aria-label="breadcrumb">
                <ol class="breadcrumb mb-0">
                    <li class="breadcrumb-item"><a href="../dashboard.php">Dashboard</a></li>
                    <li class="breadcrumb-item"><a href="index.php">Rekap SPPD</a></li>
                    <li class="breadcrumb-item active">Edit Rekap</li>
                </ol>
            </nav>
        </div>

        <?php if (isset($_SESSION['alert'])): ?>
        <div class="alert alert-<?= $_SESSION['alert']['type'] ?> alert-dismissible fade show" role="alert">
            <?= $_SESSION['alert']['message'] ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['alert']); endif; ?>

        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Form Edit Rekap SPPD</h5>
            </div>
            <div class="card-body">
                <form action="" method="POST" id="formEditRekap">
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="no_rekap" class="form-label">Nomor Rekap</label>
                                <input type="text" class="form-control" id="no_rekap" name="no_rekap" value="<?= htmlspecialchars($rekap['no_rekap']) ?>" required>
                            </div>
                            <div class="mb-3">
                                <label for="tanggal_rekap" class="form-label">Tanggal Rekap</label>
                                <input type="date" class="form-control" id="tanggal_rekap" name="tanggal_rekap" value="<?= htmlspecialchars($rekap['tanggal_rekap']) ?>" required>
                            </div>
                        </div>
                    </div>

                    <div class="table-responsive">
                        <table class="table table-hover" id="suratTable">
                            <thead>
                                <tr>
                                    <th style="width: 50px">
                                        <input type="checkbox" id="checkAll" class="form-check-input">
                                    </th>
                                    <th>Nomor Surat</th>
                                    <th>Kode Sub Kegiatan <br> <small>(kode rekening)</small></th>
                                    <th>Maksud</th>
                                    <th class="text-end">Total Biaya</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($row = mysqli_fetch_assoc($result_surat)): ?>
                                <tr>
                                    <td>
                                        <input type="checkbox" name="surat_ids[]" value="<?= $row['id_surat'] ?>" 
                                               class="form-check-input" 
                                               <?= in_array($row['id_surat'], $selected_surat_ids) ? 'checked' : '' ?>>
                                    </td>
                                    <td><?= htmlspecialchars($row['no_surat']) ?></td>
                                    <td><?= empty($row['no_sub_kegiatan']) ? '<span class="text-danger">Belum diisi</span>' : htmlspecialchars($row['no_sub_kegiatan']) ?></td>
                                    <td><?= nl2br(htmlspecialchars($row['maksud'])) ?></td>
                                    <td class="text-end">Rp <?= number_format($row['total_biaya'] ?? 0, 0, ',', '.') ?></td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>

                    <div class="mt-3">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Simpan Perubahan
                        </button>
                        <a href="detail.php?id=<?= $id_rekap ?>" class="btn btn-secondary">
                            <i class="bi bi-x-circle"></i> Batal
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.10.24/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.10.24/js/dataTables.bootstrap5.min.js"></script>

    <script>
    $(document).ready(function() {
        // Initialize DataTable
        $('#suratTable').DataTable({
            "responsive": true,
            "lengthMenu": [[10, 25, 50, -1], [10, 25, 50, "Semua"]],
            "language": {
                "url": "//cdn.datatables.net/plug-ins/1.10.24/i18n/Indonesian.json"
            }
        });

        // Handle check all
        $('#checkAll').change(function() {
            $('input[name="surat_ids[]"]').prop('checked', $(this).prop('checked'));
        });

        // Update checkAll state when individual checkboxes change
        $('input[name="surat_ids[]"]').change(function() {
            var allChecked = $('input[name="surat_ids[]"]').length === $('input[name="surat_ids[]"]:checked').length;
            $('#checkAll').prop('checked', allChecked);
        });

        // Form validation
        $('#formEditRekap').submit(function(e) {
            var checkedSurat = $('input[name="surat_ids[]"]:checked').length;
            if (checkedSurat === 0) {
                e.preventDefault();
                alert('Pilih minimal satu surat tugas');
                return false;
            }
            return true;
        });
    });
    </script>

    <style>
    /* Background dan Container */
    body {
        background: linear-gradient(135deg,rgb(231, 255, 255) 0%,rgb(236, 255, 246) 100%);
        min-height: 100vh;
    }

    .container-fluid {
        padding: 2rem;
    }

    /* Header Section */
    .header-wrapper {
        background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
        padding: 1.2rem;
        border-radius: 12px;
        margin-bottom: 1rem;
        color: white;
    }

    .header-wrapper h1 {
        font-size: 1.5rem;
        font-weight: 600;
        margin-bottom: 0.2rem;
    }

    .header-wrapper p {
        font-size: 0.9rem;
        opacity: 0.8;
        margin-bottom: 0.5rem;
    }

    .breadcrumb {
        background: rgba(255, 255, 255, 0.1);
        padding: 0.5rem 0.8rem;
        border-radius: 6px;
        margin: 0;
    }

    .breadcrumb-item a {
        color: rgba(255, 255, 255, 0.8);
        text-decoration: none;
    }

    .breadcrumb-item.active {
        color: white;
    }

    /* Card Styling */
    .card {
        background: rgba(255, 255, 255, 0.95);
        border: none;
        border-radius: 15px;
        box-shadow: 0 8px 32px 0 rgba(14, 165, 233, 0.15);
    }

    .card-header {
        background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
        color: white;
        border-radius: 15px 15px 0 0 !important;
        border: none;
    }

    /* Form Controls */
    .form-control {
        border-radius: 8px;
        border: 1px solid rgba(14, 165, 233, 0.2);
        padding: 0.6rem 1rem;
    }

    .form-control:focus {
        border-color: #0ea5e9;
        box-shadow: 0 0 0 0.2rem rgba(14, 165, 233, 0.25);
    }

    /* Table Styling */
    .table {
        margin: 0;
    }

    .table thead th {
        background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
        color: white;
        font-weight: 600;
        border: none;
    }

    .table tbody td {
        vertical-align: middle;
    }

    /* Buttons */
    .btn-primary {
        background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
        border: none;
        padding: 0.5rem 1rem;
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
    }

    .btn-primary:hover {
        background: linear-gradient(135deg, #0891ce 0%, #20b4a3 100%);
        transform: translateY(-1px);
    }

    .btn-secondary {
        background: #64748b;
        border: none;
    }

    /* Responsive */
    @media (max-width: 768px) {
        .container-fluid {
            padding: 1rem;
        }

        .header-wrapper {
            padding: 1rem;
        }

        .card-body {
            padding: 1rem;
        }
    }
    </style>
</body>
</html> 