<?php
session_start();
require_once '../config/database.php';
require_once '../config/config.php';

if (isset($_GET['id'])) {
    $id_surat = $_GET['id'];
    
    // Query untuk mengambil data surat tugas
    $query_surat = "SELECT st.*, 
                           GROUP_CONCAT(DISTINCT p.nama_pegawai ORDER BY p.nama_pegawai SEPARATOR ', ') as nama_pegawai,
                           COUNT(DISTINCT p.id_pegawai) as jumlah_pegawai,
                           DATEDIFF(tanggal_kembali, tanggal_berangkat) + 1 as lama_perjalanan
                    FROM surat_tugas st 
                    LEFT JOIN detail_surat_tugas dst ON st.id_surat = dst.id_surat
                    LEFT JOIN pegawai p ON dst.id_pegawai = p.id_pegawai 
                    WHERE st.id_surat = ?
                    GROUP BY st.id_surat";
    
    $stmt = mysqli_prepare($conn, $query_surat);
    mysqli_stmt_bind_param($stmt, "i", $id_surat);
    mysqli_stmt_execute($stmt);
    $result_surat = mysqli_stmt_get_result($stmt);
    $surat = mysqli_fetch_assoc($result_surat);

    // Query untuk mengambil rincian biaya dari tabel rincian_bimtek_perorangan
    $query_rincian = "SELECT rbp.*, p.nama_pegawai 
                      FROM rincian_bimtek_perorangan rbp
                      JOIN pegawai p ON rbp.id_pegawai = p.id_pegawai
                      WHERE rbp.id_surat = ?";
    $stmt = mysqli_prepare($conn, $query_rincian);
    mysqli_stmt_bind_param($stmt, "i", $id_surat);
    mysqli_stmt_execute($stmt);
    $result_rincian = mysqli_stmt_get_result($stmt);

    // Query untuk menghitung total biaya per komponen
    $query_total = "SELECT 
                    SUM(uang_harian_per_hari) as total_uang_harian_per_hari,
                    SUM(total_uang_harian) as total_uang_harian,
                    SUM(representatif_per_hari) as total_representatif_per_hari,
                    SUM(total_representatif) as total_representatif,
                    SUM(uang_diklat_per_hari) as total_uang_diklat_per_hari,
                    SUM(total_uang_diklat) as total_uang_diklat,
                    SUM(transport_dalam_provinsi) as total_transport_dalam,
                    SUM(transport_luar_provinsi) as total_transport_luar,
                    SUM(biaya_pesawat_berangkat) as total_pesawat_berangkat,
                    SUM(biaya_pesawat_pulang) as total_pesawat_pulang,
                    SUM(akomodasi) as total_akomodasi,
                    SUM(total_biaya) as grand_total
                    FROM rincian_bimtek_perorangan
                    WHERE id_surat = ?";
    $stmt = mysqli_prepare($conn, $query_total);
    mysqli_stmt_bind_param($stmt, "i", $id_surat);
    mysqli_stmt_execute($stmt);
    $result_total = mysqli_stmt_get_result($stmt);
    $total_biaya = mysqli_fetch_assoc($result_total);

    // Query untuk mengambil data rincian riil
    $query_rincian_riil = "SELECT r.*, p.nama_pegawai 
                          FROM rincian_biaya_riil r
                          JOIN pegawai p ON r.id_pegawai = p.id_pegawai 
                          WHERE r.id_surat = ?
                          ORDER BY p.nama_pegawai";
    $stmt = $conn->prepare($query_rincian_riil);
    $stmt->bind_param("i", $id_surat);
    $stmt->execute();
    $result_rincian_riil = $stmt->get_result();

    $rincian_riil_by_pegawai = [];
    while ($row = $result_rincian_riil->fetch_assoc()) {
        $rincian_riil_by_pegawai[$row['id_pegawai']][] = $row;
    }

    // Query untuk mengambil semua pegawai untuk surat ini (untuk tombol Edit Semua)
    $query_pegawai = "SELECT DISTINCT p.id_pegawai, p.nama_pegawai 
                    FROM detail_surat_tugas dst 
                    JOIN pegawai p ON dst.id_pegawai = p.id_pegawai 
                    WHERE dst.id_surat = ?
                    ORDER BY p.nama_pegawai";
    $stmt = $conn->prepare($query_pegawai);
    $stmt->bind_param("i", $id_surat);
    $stmt->execute();
    $result_pegawai = $stmt->get_result();
    $all_pegawai = $result_pegawai->fetch_all(MYSQLI_ASSOC);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Detail Rincian Biaya Reguler - SPPD</title>
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyNCAyNCI+PHBhdGggZmlsbD0iIzRlNzNkZiIgZD0iTTIwLDhINFY2aDIwdjJIMjB6IE0xNSwxMEg0djJoMTFWMTB6IE00LDE2aDd2Mkg0VjE2eiBNMjEuNywxMy4zTDIwLjMsMTJsMS40LTEuNEMyMi4xLDEwLjIsMjIuMSw5LjgsMjEuNyw5LjRsLTEuMS0xLjFDMjAuMiw3LjksMTkuOCw3LjksMTkuNCw4LjNMMTgsMTBsLTEuNC0xLjRjLTAuNC0wLjQtMS0wLjQtMS40LDBsLTcuNSw3LjVjLTAuMiwwLjItMC4zLDAuNC0wLjMsMC43djIuOWMwLDAuNSwwLjQsMC45LDAuOSwwLjloMi45YzAuMywwLDAuNS0wLjEsMC43LTAuM2w3LjUtNy41QzIyLjEsMTQuMywyMi4xLDEzLjcsMjEuNywxMy4zeiIvPjwvc3ZnPg==">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg,rgb(231, 255, 255) 0%,rgb(236, 255, 246) 100%);
            min-height: 100vh;
            padding: 0;
            margin: 0;
            color: #2c3e50;
        }

        /* Card Styling */
        .card {
            background: rgba(255, 255, 255, 0.95);
            border: none;
            border-radius: 0;
            box-shadow: 0 8px 32px 0 rgba(31, 38, 135, 0.15);
            backdrop-filter: blur(8px);
            margin-bottom: 0;
            transition: transform 0.3s ease;
        }

        .card:hover {
            transform: translateY(-5px);
        }

        .card-header {
            border-radius: 0 !important;
            padding: 20px;
            border: none;
        }

        .card-header h5 {
            font-size: 1.25rem;
            font-weight: 600;
            margin: 0;
        }

        .card-body {
            padding: 25px;
        }

        /* Header Section */
        .header-section {
            background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
            color: white;
            padding: 25px;
            margin-bottom: 0;
            border-radius: 0;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }

        .header-section h2 {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 10px;
        }

        /* Breadcrumb */
        .breadcrumb {
            background: rgba(255, 255, 255, 0.1);
            padding: 10px 20px;
            border-radius: 8px;
            margin: 0;
        }

        .breadcrumb-item a {
            color: rgba(255, 255, 255, 0.8);
            text-decoration: none;
            transition: color 0.3s ease;
        }

        .breadcrumb-item a:hover {
            color: white;
        }

        /* Table Styling */
        .table {
            margin: 0;
        }

        .table th {
            background-color: #f8f9fa;
            color: #4e73df;
            font-weight: 600;
            padding: 15px;
            border-bottom: 2px solid #e3e6f0;
        }

        .table td {
            padding: 15px;
            vertical-align: middle;
            border-color: #e3e6f0;
        }

        .table-hover tbody tr:hover {
            background-color: rgba(78, 115, 223, 0.05);
        }

        .table-borderless td {
            border: none;
        }

        /* Button Styling */
        .btn {
            padding: 10px 20px;
            border-radius: 8px;
            font-weight: 500;
            transition: all 0.3s ease;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }

        .btn-light {
            background: rgba(255, 255, 255, 0.9);
            border: none;
        }

        .btn-primary {
            background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
            border: none;
        }

        .btn-success {
            background: linear-gradient(135deg, #1cc88a 0%, #13855c 100%);
            border: none;
        }

        .btn-warning {
            background: linear-gradient(135deg, #f6c23e 0%, #dda20a 100%);
            border: none;
        }

        .btn-info {
            background: linear-gradient(135deg, #36b9cc 0%, #258391 100%);
            border: none;
        }

        /* Alert Styling */
        .alert {
            border-radius: 10px;
            border: none;
            padding: 20px;
        }

        .alert-info {
            background: rgba(54, 185, 204, 0.1);
            color: #36b9cc;
            border-left: 4px solid #36b9cc;
        }

        /* Table Header Colors */
        .bg-primary {
            background: linear-gradient(135deg, #4e73df 0%, #224abe 100%) !important;
        }

        .bg-success {
            background: linear-gradient(135deg, #1cc88a 0%, #13855c 100%) !important;
        }

        /* Total Row Styling */
        .table-primary {
            background: rgba(78, 115, 223, 0.1) !important;
        }

        .text-muted {
            color: #858796 !important;
        }

        /* Responsive Table */
        .table-responsive {
            border-radius: 10px;
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.05);
        }

        /* Print Buttons Container */
        .d-flex.gap-2 {
            background: rgba(255, 255, 255, 0.95);
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
        }

        /* Icons */
        .bi {
            margin-right: 5px;
        }

        /* Animation */
        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .card {
            animation: fadeIn 0.5s ease-out;
        }

        /* Small Text */
        small.text-muted {
            font-size: 0.85rem;
            opacity: 0.8;
        }

        .modal-content {
            border-radius: 10px;
            border: none;
            box-shadow: 0 2px 20px rgba(0,0,0,0.1);
        }
        .modal-header {
            padding: 20px 20px 10px;
        }
        .modal-title {
            font-size: 18px;
            font-weight: 500;
        }
        .modal-body {
            padding: 20px;
        }
        .form-label {
            font-size: 14px;
            margin-bottom: 8px;
        }
        .form-control {
            padding: 10px 12px;
            border: 1px solid #E0E0E0;
            border-radius: 6px;
            font-size: 14px;
        }
        .form-control:focus {
            border-color: #4461F2;
            box-shadow: none;
        }
        .btn-link {
            text-decoration: none;
            font-weight: 500;
        }
        .btn-link:hover {
            text-decoration: none;
            opacity: 0.8;
        }

        .content-wrapper {
            padding: 20px;
        }

        @media (min-width: 768px) {
            .content-wrapper {
                padding: 30px;
            }
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .row.g-3 > [class*='col-'] {
                padding: 10px;
            }
        }

        .nav-pills {
            background: #fff;
            padding: 15px;
            border-radius: 10px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        .nav-pills .nav-link {
            color: #495057;
            padding: 10px 20px;
            border-radius: 8px;
            transition: all 0.3s ease;
            margin-right: 10px;
        }
        .nav-pills .nav-link:hover {
            background-color: #e9ecef;
        }
        .nav-pills .nav-link.active {
            background-color: #4e73df;
            color: #fff;
        }
        .nav-pills .nav-link i {
            margin-right: 5px;
        }
        .tab-content {
            background: transparent;
        }
        .tab-pane {
            transition: all 0.3s ease;
        }
    </style>
</head>
<body>
        <div class="header-section">
            <h2 class="mb-0">Detail Rincian Biaya Reguler</h2>
            <nav aria-label="breadcrumb">
                <ol class="breadcrumb mb-0">
                    <li class="breadcrumb-item"><a href="../dashboard.php" class="text-white">Dashboard</a></li>
                    <li class="breadcrumb-item"><a href="index.php" class="text-white">Rincian Biaya Reguler</a></li>
                    <li class="breadcrumb-item active text-white" aria-current="page">Detail</li>
                </ol>
            </nav>
        </div>

    <div class="content-wrapper">
        <?php if (isset($surat)): ?>
        <!-- Informasi Surat Tugas -->
        <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Informasi Surat Tugas</h5>
                    <button type="button" class="btn btn-light btn-sm" onclick="inputSubKegiatan(<?= $surat['id_surat'] ?>)">
                        <i class="bi bi-plus-circle"></i> Input Sub Kegiatan
                    </button>
                </div>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <table class="table table-borderless">
                            <tr>
                                <th>ID Surat</th>
                                <td><?= htmlspecialchars($surat['id_surat'], ENT_QUOTES, 'UTF-8') ?></td>
                            </tr>
                            <tr>
                                <th>Jenis Surat</th>
                                <td><?= htmlspecialchars($surat['jenis_surat'], ENT_QUOTES, 'UTF-8') ?></td>
                            </tr>
                            <tr>
                                <th>No. Surat</th>
                                <td><?= htmlspecialchars($surat['no_surat'], ENT_QUOTES, 'UTF-8') ?></td>
                            </tr>
                            <tr>
                                <th>No. SPPD</th>
                                <td><?= htmlspecialchars($surat['no_sppd'], ENT_QUOTES, 'UTF-8') ?></td>
                            </tr>
                            <tr>
                                <th>No.Sub Kegiatan & Kode Rekening</th>
                                <td><?= htmlspecialchars($surat['no_sub_kegiatan'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
                            </tr>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <table class="table table-borderless">
                            <tr>
                                <th>Nama Pegawai</th>
                                <td><?= htmlspecialchars($surat['nama_pegawai'], ENT_QUOTES, 'UTF-8') ?></td>
                            </tr>
                            <tr>
                                <th>Tanggal Surat</th>
                                <td><?= date('d/m/Y', strtotime($surat['tanggal_surat'])) ?></td>
                            </tr>
                            <tr>
                                <th>Tanggal Berangkat</th>
                                <td><?= date('d/m/Y', strtotime($surat['tanggal_berangkat'])) ?></td>
                            </tr>
                            <tr>
                                <th>Tanggal Kembali</th>
                                <td><?= date('d/m/Y', strtotime($surat['tanggal_kembali'])) ?></td>
                            </tr>
                            <tr>
                                <th>Lama Perjalanan</th>
                                <td><?= $surat['lama_perjalanan'] ?> Hari</td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Navigation Tabs -->
        <div class="mb-4">
            <ul class="nav nav-pills" id="rincianTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="rincian-biaya-tab" data-bs-toggle="pill" 
                            data-bs-target="#rincian-biaya" type="button" role="tab">
                        <i class="bi bi-calculator"></i> Rincian Biaya
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="rincian-riil-tab" data-bs-toggle="pill" 
                            data-bs-target="#rincian-riil" type="button" role="tab">
                        <i class="bi bi-receipt"></i> Rincian Biaya Riil
                    </button>
                </li>
            </ul>
        </div>

        <!-- Tab Content -->
        <div class="tab-content" id="rincianTabContent">
            <!-- Rincian Biaya Tab -->
            <div class="tab-pane fade show active" id="rincian-biaya" role="tabpanel">
                <!-- Rincian Biaya Card -->
        <div class="card">
            <div class="card-header bg-success text-white d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Rincian Biaya</h5>
                <div>
                            <?php if (mysqli_num_rows($result_rincian) > 0): ?>
                                <a href="javascript:void(0)" class="btn btn-danger btn-sm me-2" onclick="confirmDelete(<?= $id_surat ?>)">
                        <i class="bi bi-trash"></i> Hapus Rincian
                    </a>
                    <a href="tambah_rincian.php?id=<?= $id_surat ?>" class="btn btn-light btn-sm">
                                    <i class="bi bi-pencil"></i> Edit Rincian
                    </a>
                            <?php else: ?>
                                <a href="tambah_rincian.php?id=<?= $id_surat ?>" class="btn btn-light btn-sm">
                                    <i class="bi bi-plus-circle"></i> Tambah Rincian
                                </a>
                            <?php endif; ?>
                </div>
            </div>
            <div class="card-body">
                        <!-- Search Form untuk Rincian Biaya -->
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="input-group">
                                    <span class="input-group-text">
                                        <i class="bi bi-search"></i>
                                    </span>
                                    <input type="text" 
                                           class="form-control" 
                                           id="searchRincianBiaya" 
                                           placeholder="Cari nama pegawai..."
                                           onkeyup="filterRincianBiaya()">
                                </div>
                            </div>
                        </div>

                        <?php if (mysqli_num_rows($result_rincian) > 0): ?>
                            <?php while ($rincian = mysqli_fetch_assoc($result_rincian)): ?>
                                <div class="card mb-3 shadow-sm rincian-biaya-card" data-nama="<?= strtolower(htmlspecialchars($rincian['nama_pegawai'])) ?>">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between align-items-center mb-3">
                                            <h6 class="card-title mb-0"><?= htmlspecialchars($rincian['nama_pegawai'], ENT_QUOTES, 'UTF-8') ?></h6>
                                            <a href="edit_rincian_perorangan.php?id=<?= $rincian['id_rincian_perorangan'] ?>&surat=<?= $id_surat ?>" 
                                               class="btn btn-sm btn-warning">
                                                <i class="bi bi-pencil"></i> Edit
                                            </a>
                                        </div>
                                        <div class="row g-3">
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Uang Harian</small>
                                                    <span class="fw-bold">Rp <?= number_format($rincian['total_uang_harian'], 0, ',', '.') ?></span>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Representatif</small>
                                                    <span class="fw-bold">Rp <?= number_format($rincian['total_representatif'] ?? 0, 0, ',', '.') ?></span>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Uang Diklat</small>
                                                    <span class="fw-bold">Rp <?= number_format($rincian['total_uang_diklat'], 0, ',', '.') ?></span>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Transport Dalam Provinsi</small>
                                                    <span class="fw-bold">Rp <?= number_format($rincian['transport_dalam_provinsi'], 0, ',', '.') ?></span>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Transport Luar Provinsi</small>
                                                    <span class="fw-bold">Rp <?= number_format($rincian['transport_luar_provinsi'], 0, ',', '.') ?></span>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Biaya Pesawat</small>
                                                    <span class="fw-bold">Rp <?= number_format($rincian['biaya_pesawat_berangkat'] + $rincian['biaya_pesawat_pulang'], 0, ',', '.') ?></span>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Akomodasi</small>
                                                    <span class="fw-bold">Rp <?= number_format($rincian['akomodasi'], 0, ',', '.') ?></span>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Jarak</small>
                                                    <span class="fw-bold"><?= $rincian['jarak'] ?> KM</span>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Biaya Jarak</small>
                                                    <span class="fw-bold">Rp <?= number_format($rincian['biaya_jarak'], 0, ',', '.') ?></span>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="d-flex flex-column">
                                                    <small class="text-muted">Total</small>
                                                    <span class="fw-bold text-primary">Rp <?= number_format($rincian['total_biaya'], 0, ',', '.') ?></span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endwhile; ?>

                            <!-- Total Keseluruhan dengan Rincian -->
                            <div class="card mt-4">
                                <div class="card-header bg-primary text-white">
                                    <h5 class="mb-0">Total Keseluruhan</h5>
                                </div>
                                <div class="card-body">
                                    <?php
                                    $total_query = "SELECT 
                                        SUM(total_uang_harian) as total_uang_harian,
                                        SUM(total_representatif) as total_representatif,
                                        SUM(total_uang_diklat) as total_uang_diklat,
                                        SUM(transport_dalam_provinsi) as total_transport_dalam,
                                        SUM(transport_luar_provinsi) as total_transport_luar,
                                        SUM(biaya_pesawat_berangkat + biaya_pesawat_pulang) as total_pesawat,
                                        SUM(akomodasi) as total_akomodasi,
                                        SUM(biaya_jarak) as total_biaya_jarak,
                                        SUM(total_biaya) as grand_total
                                        FROM rincian_bimtek_perorangan
                                        WHERE id_surat = ?";
                                    $stmt = $conn->prepare($total_query);
                                    $stmt->bind_param("i", $id_surat);
                                    $stmt->execute();
                                    $result_total = $stmt->get_result();
                                    $total_row = $result_total->fetch_assoc();
                                    ?>
                                    
                                    <div class="table-responsive">
                                        <table class="table table-borderless">
                                            <tbody>
                                                <tr>
                                                    <td style="width: 200px;">
                                                        <span class="text-muted">Uang Harian</span>
                                                    </td>
                                                    <td class="text-end" style="width: 200px;">
                                                        <span class="fw-bold">Rp <?= number_format($total_row['total_uang_harian'], 0, ',', '.') ?></span>
                                                    </td>
                                                    <td style="width: 200px;">
                                                        <span class="text-muted">Transport Dalam</span>
                                                    </td>
                                                    <td class="text-end" style="width: 200px;">
                                                        <span class="fw-bold">Rp <?= number_format($total_row['total_transport_dalam'], 0, ',', '.') ?></span>
                                                    </td>
                                                    </tr>
                                                    <tr>
                                                    <td>
                                                        <span class="text-muted">Representatif</span>
                                                         </td>
                                                        <td class="text-end">
                                                        <span class="fw-bold">Rp <?= number_format($total_row['total_representatif'] ?? 0, 0, ',', '.') ?></span>
                                                    </td>
                                                    <td>
                                                        <span class="text-muted">Transport Luar</span>
                                                    </td>
                                                    <td class="text-end">
                                                        <span class="fw-bold">Rp <?= number_format($total_row['total_transport_luar'], 0, ',', '.') ?></span>
                                                        </td>
                                                    </tr>
                                                    <tr>
                                                        <td>
                                                        <span class="text-muted">Uang Diklat</span>
                                                  </td>
                                                    <td class="text-end">
                                                        <span class="fw-bold">Rp <?= number_format($total_row['total_uang_diklat'], 0, ',', '.') ?></span>
                                                    </td>
                                                    <td>
                                                        <span class="text-muted">Biaya Pesawat</span>
                                                    </td>
                                                    <td class="text-end">
                                                        <span class="fw-bold">Rp <?= number_format($total_row['total_pesawat'], 0, ',', '.') ?></span>
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td>
                                                        <span class="text-muted">Akomodasi</span>
                        </td>
                                                    <td class="text-end">
                                                        <span class="fw-bold">Rp <?= number_format($total_row['total_akomodasi'], 0, ',', '.') ?></span>
                                                    </td>
                                                    <td>
                                                        <span class="text-muted">Biaya Jarak</span>
                                                    </td>
                                                    <td class="text-end">
                                                        <span class="fw-bold">Rp <?= number_format($total_row['total_biaya_jarak'], 0, ',', '.') ?></span>
                                                    </td>
                                                </tr>
                                            </tbody>
                                            <tfoot>
                                                <tr class="border-top">
                                                    <td colspan="3" class="pt-3">
                                                        <h6 class="mb-0">Total Keseluruhan</h6>
                                                    </td>
                                                    <td class="text-end pt-3">
                                                        <h5 class="mb-0 text-primary">Rp <?= number_format($total_row['grand_total'], 0, ',', '.') ?></h5>
                        </td>
                    </tr>
                                            </tfoot>
                                        </table>
                                    </div>

                                    <div class="mt-2">
                            <small class="text-muted">
                                            <i class="bi bi-info-circle"></i> 
                                            Total dari seluruh rincian biaya pegawai
                            </small>
                                    </div>
                                </div>
                            </div>
                <?php else: ?>
                <div class="alert alert-info mb-0">
                    <i class="bi bi-info-circle"></i> Belum ada rincian biaya.
                    <br>
                    <a href="tambah_rincian.php?id=<?= $id_surat ?>" class="btn btn-primary mt-2">
                        <i class="bi bi-plus-circle"></i> Tambah Rincian
                    </a>
                </div>
                <?php endif; ?>
                    </div>
            </div>
        </div>

            <!-- Rincian Riil Tab -->
            <div class="tab-pane fade" id="rincian-riil" role="tabpanel">
                <div class="alert alert-info mb-3">
                    <i class="bi bi-info-circle"></i> 
                    <?php if (empty($rincian_riil_by_pegawai)): ?>
                        Belum ada rincian biaya riil.
                    <?php else: ?>
                        Rincian biaya riil sudah ada.
                    <?php endif; ?>
                    <div class="mt-3">
                        <?php if (!empty($rincian_riil_by_pegawai)): ?>
                            <button type="button" 
                                    class="btn btn-danger me-2"
                                    onclick="confirmDeleteRincianRiil(<?= htmlspecialchars(json_encode($id_surat)) ?>)">
                                <i class="bi bi-trash"></i> Hapus Rincian Biaya Riil
                            </button>
                        <?php endif; ?>
                        <button type="button" 
                                class="btn btn-primary"
                                onclick='editAllRincianRiil(<?= htmlspecialchars(json_encode($id_surat)) ?>, <?= htmlspecialchars(json_encode($all_pegawai)) ?>)'>
                            <i class="bi bi-<?= empty($rincian_riil_by_pegawai) ? 'plus-circle' : 'pencil' ?>"></i> 
                            <?= empty($rincian_riil_by_pegawai) ? 'Tambah' : 'Edit' ?> Rincian Biaya Riil
                        </button>
                    </div>
                </div>

                <?php if (!empty($rincian_riil_by_pegawai)): ?>
                <div class="row" id="rincianRiilContainer">
                    <?php foreach ($rincian_riil_by_pegawai as $id_pegawai => $rincian_list): 
                        $nama_pegawai = $rincian_list[0]['nama_pegawai'];
                        $total_pegawai = 0;
                    ?>
                    <div class="col-md-6 mb-4 rincian-riil-card" data-nama="<?= strtolower(htmlspecialchars($nama_pegawai)) ?>">
                        <div class="card h-100">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <h5 class="card-title mb-0"><?= htmlspecialchars($nama_pegawai) ?></h5>
                                <div>
                                    <button type="button" 
                                            class="btn btn-primary"
                                            onclick='editRincianRiil(<?= htmlspecialchars(json_encode($id_surat)) ?>, 
                                                        <?= htmlspecialchars(json_encode($id_pegawai)) ?>, 
                                                        <?= htmlspecialchars(json_encode($nama_pegawai)) ?>, 
                                                        <?= htmlspecialchars(json_encode($rincian_list)) ?>)'>
                                        <i class="bi bi-pencil"></i> Edit
                                    </button>
                                </div>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Uraian</th>
                                                <th class="text-end">Jumlah</th>
                                                <th>Keterangan</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($rincian_list as $rincian): 
                                                $total_pegawai += $rincian['jumlah'];
                                            ?>
                                            <tr>
                                                <td><?= htmlspecialchars($rincian['uraian']) ?></td>
                                                <td class="text-end">Rp <?= number_format($rincian['jumlah'], 0, ',', '.') ?></td>
                                                <td><?= htmlspecialchars($rincian['keterangan']) ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                        <tfoot>
                                            <tr>
                                                <th>Total</th>
                                                <th class="text-end">Rp <?= number_format($total_pegawai, 0, ',', '.') ?></th>
                                                <th></th>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        </div>

        <?php else: ?>
        <div class="alert alert-danger">
            Data surat tugas tidak ditemukan.
        </div>
        <?php endif; ?>
    </div>

    <!-- Modal Input Sub Kegiatan -->
    <div class="modal fade" id="subKegiatanModal" tabindex="-1" aria-labelledby="subKegiatanModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header border-0">
                    <h5 class="modal-title" id="subKegiatanModalLabel">Input Nomor Sub Kegiatan & Kode Rekening</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body pt-0">
                    <form id="subKegiatanForm" onsubmit="saveSubKegiatan(event)">
                        <input type="hidden" id="id_surat" name="id_surat" value="<?= $surat['id_surat'] ?>">
                        <div class="mb-3">
                            <label for="no_sub_kegiatan" class="form-label"><small class="text-muted">sesuaikan dengan format yang ada</small></label>
                            <input type="text" class="form-control" id="no_sub_kegiatan" name="no_sub_kegiatan" 
                                   value="0.00.00.0.00.0000.0.0.00.00.00.0000" required>
                        </div>
                        <div class="text-end mt-4">
                            <button type="button" class="btn btn-link text-dark me-2" data-bs-dismiss="modal">Batal</button>
                            <button type="submit" class="btn text-white" style="background-color: #4461F2;">Simpan</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>



    <!-- Modal Edit Rincian Riil -->
    <div class="modal fade" id="editRincianRiilModal" tabindex="-1" aria-labelledby="editRincianRiilModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header">
                    <div>
                        <h5 class="modal-title" id="editRincianRiilModalLabel">Edit Rincian Biaya Riil</h5>
                        <p class="mb-0 text-muted" id="edit_riil_nama_pegawai"></p>
                    </div>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="formEditRincianRiil" onsubmit="saveRincianRiil(event)">
                        <input type="hidden" id="edit_riil_id_surat" name="id_surat">
                        <div id="rincianRiilItems">
                            <!-- Rincian items will be added here dynamically -->
                        </div>
                        <div class="text-end mt-4">
                            <button type="button" class="btn btn-secondary me-2" data-bs-dismiss="modal">Batal</button>
                            <button type="submit" class="btn btn-primary">Simpan Perubahan</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.11.8/dist/umd/popper.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script>
    function inputSubKegiatan(id_surat) {
        document.getElementById('id_surat').value = id_surat;
        var subKegiatanModal = new bootstrap.Modal(document.getElementById('subKegiatanModal'));
        subKegiatanModal.show();
    }

    function saveSubKegiatan(event) {
        event.preventDefault();
        
        var formData = new FormData(document.getElementById('subKegiatanForm'));
        
        // Debug: Log form data
        console.log('Form data being sent:');
        for (var pair of formData.entries()) {
            console.log(pair[0] + ': ' + pair[1]);
        }
        
        fetch('save_sub_kegiatan.php', {
            method: 'POST',
            body: formData
        })
        .then(response => {
            console.log('Response status:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('Response data:', data);
            
            var modal = bootstrap.Modal.getInstance(document.getElementById('subKegiatanModal'));
            modal.hide();
            
            if (data.success) {
                Swal.fire({
                    icon: 'success',
                    title: 'Berhasil',
                    text: 'Nomor sub kegiatan berhasil disimpan',
                    showConfirmButton: false,
                    timer: 1500
                }).then(() => {
                    // Force reload from server, not from cache
                    window.location.reload(true);
                });
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Gagal',
                    text: data.message || 'Terjadi kesalahan saat menyimpan data'
                });
            }
        })
        .catch(error => {
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Terjadi kesalahan pada server'
            });
        });
    }

    // Format input uang
    $('.money').on('input', function() {
        let value = $(this).val().replace(/[^\d]/g, '');
        $(this).val(new Intl.NumberFormat('id-ID').format(value));
    });

    // Fungsi format rupiah
    function formatRupiah(angka) {
        if (!angka) return '0';
        return new Intl.NumberFormat('id-ID').format(angka);
    }





    // Simpan tab yang aktif ke localStorage saat tab berubah
    document.querySelectorAll('#rincianTabs .nav-link').forEach(tab => {
        tab.addEventListener('shown.bs.tab', function (e) {
            localStorage.setItem('activeRincianTab', e.target.id);
        });
    });

    // Cek dan set tab aktif saat halaman dimuat
    document.addEventListener('DOMContentLoaded', function() {
        const activeTab = localStorage.getItem('activeRincianTab');
        if (activeTab) {
            const tab = document.querySelector('#' + activeTab);
            if (tab) {
                const bsTab = new bootstrap.Tab(tab);
                bsTab.show();
            }
        }
    });

    function editRincianRiil(idSurat, idPegawai, namaPegawai, rincianList) {
        // Set data pada modal
        document.getElementById('edit_riil_nama_pegawai').textContent = namaPegawai;
        document.getElementById('edit_riil_id_surat').value = idSurat;
        
        // Reset container rincian items
        const container = document.getElementById('rincianRiilItems');
        container.innerHTML = '';

        // Tambahkan hidden input untuk id_pegawai
        const hiddenInput = document.createElement('input');
        hiddenInput.type = 'hidden';
        hiddenInput.name = 'id_pegawai[]';
        hiddenInput.value = idPegawai;
        container.appendChild(hiddenInput);
        
        // Tambahkan item rincian yang ada
        if (rincianList && rincianList.length > 0) {
            rincianList.forEach((item, index) => {
                const itemHtml = `
                    <div class="rincian-item mb-3">
                        <div class="card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h6 class="card-title mb-0">Item #${index + 1}</h6>
                                    ${index > 0 ? `
                                        <button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('.rincian-item').remove()">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    ` : ''}
                                </div>
                                <div class="row g-3">
                                    <div class="col-md-12">
                                        <label class="form-label">Uraian</label>
                                        <input type="text" class="form-control" name="uraian[]" value="${item.uraian}" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Jumlah</label>
                                        <div class="input-group">
                                            <span class="input-group-text">Rp</span>
                                            <input type="text" class="form-control money" name="jumlah[]" value="${formatRupiah(item.jumlah)}" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Keterangan</label>
                                        <input type="text" class="form-control" name="keterangan[]" value="${item.keterangan || ''}">
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
                container.insertAdjacentHTML('beforeend', itemHtml);
            });
        } else {
            // Tambahkan satu item kosong jika tidak ada data
            const itemHtml = `
                <div class="rincian-item mb-3">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center mb-3">
                                <h6 class="card-title mb-0">Item #1</h6>
                            </div>
                            <div class="row g-3">
                                <div class="col-md-12">
                                    <label class="form-label">Uraian</label>
                                    <input type="text" class="form-control" name="uraian[]" required>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Jumlah</label>
                                    <div class="input-group">
                                        <span class="input-group-text">Rp</span>
                                        <input type="text" class="form-control money" name="jumlah[]" required>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Keterangan</label>
                                    <input type="text" class="form-control" name="keterangan[]">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            container.insertAdjacentHTML('beforeend', itemHtml);
        }

        // Tambahkan tombol Tambah Item
        const addButtonHtml = `
            <div class="text-end mt-3">
                <button type="button" class="btn btn-success" onclick="addRincianItem()">
                    <i class="bi bi-plus-circle"></i> Tambah Item
                </button>
            </div>
        `;
        container.insertAdjacentHTML('beforeend', addButtonHtml);
        
        // Initialize money format
        initializeMoneyFormat();
        
        // Tampilkan modal
        const modal = new bootstrap.Modal(document.getElementById('editRincianRiilModal'));
        modal.show();
    }

    function addRincianItem() {
        const container = document.getElementById('rincianRiilItems');
        const itemCount = document.querySelectorAll('.rincian-item').length + 1;
        
        const itemHtml = `
            <div class="rincian-item mb-3">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="card-title mb-0">Item #${itemCount}</h6>
                            <button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('.rincian-item').remove()">
                                <i class="bi bi-trash"></i>
                            </button>
                        </div>
                        <div class="row g-3">
                            <div class="col-md-12">
                                <label class="form-label">Uraian</label>
                                <input type="text" class="form-control" name="uraian[]" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Jumlah</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rp</span>
                                    <input type="text" class="form-control money" name="jumlah[]" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Keterangan</label>
                                <input type="text" class="form-control" name="keterangan[]">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Insert before the "Tambah Item" button
        const addButton = container.querySelector('.text-end');
        if (addButton) {
            addButton.insertAdjacentHTML('beforebegin', itemHtml);
        } else {
            container.insertAdjacentHTML('beforeend', itemHtml);
        }
        
        initializeMoneyFormat();
    }

    function initializeMoneyFormat() {
        document.querySelectorAll('.money').forEach(input => {
            input.addEventListener('input', function() {
                let value = this.value.replace(/[^\d]/g, '');
                this.value = formatRupiah(value);
            });
        });
    }

    function saveRincianRiil(event) {
        event.preventDefault();
        
        // Ambil semua data form
        const formData = new FormData(document.getElementById('formEditRincianRiil'));
        
        // Konversi nilai rupiah ke angka untuk jumlah
        document.querySelectorAll('input[name="jumlah[]"]').forEach(input => {
            const value = input.value.replace(/[^\d]/g, '');
            formData.append('jumlah_clean[]', value);
        });

        // Kirim data ke server
        fetch('update_rincian_riil.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Tutup modal
                bootstrap.Modal.getInstance(document.getElementById('editRincianRiilModal')).hide();
                
                // Tampilkan pesan sukses
                Swal.fire({
                    icon: 'success',
                    title: 'Berhasil',
                    text: 'Data rincian riil berhasil disimpan',
                    showConfirmButton: false,
                    timer: 1500
                }).then(() => {
                    // Refresh halaman
                    location.reload();
                });
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Gagal',
                    text: data.message || 'Terjadi kesalahan saat menyimpan data'
                });
            }
        })
        .catch(error => {
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Terjadi kesalahan pada server'
            });
        });
    }

    function editAllRincianRiil(idSurat, pegawaiList) {
        // Set data pada modal
        document.getElementById('edit_riil_nama_pegawai').textContent = 'Semua Pegawai';
        document.getElementById('edit_riil_id_surat').value = idSurat;
        
        // Reset container rincian items
        const container = document.getElementById('rincianRiilItems');
        container.innerHTML = '';

        // Tambahkan hidden input untuk setiap id_pegawai
        if (Array.isArray(pegawaiList)) {
            pegawaiList.forEach(pegawai => {
                const hiddenInput = document.createElement('input');
                hiddenInput.type = 'hidden';
                hiddenInput.name = 'id_pegawai[]';
                hiddenInput.value = pegawai.id_pegawai;
                container.appendChild(hiddenInput);
            });
        }

        // Tambahkan satu item kosong
        const itemHtml = `
            <div class="rincian-item mb-3">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="card-title mb-0">Item #1</h6>
                        </div>
                        <div class="row g-3">
                            <div class="col-md-12">
                                <label class="form-label">Uraian</label>
                                <input type="text" class="form-control" name="uraian[]" required>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Jumlah</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rp</span>
                                    <input type="text" class="form-control money" name="jumlah[]" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Keterangan</label>
                                <input type="text" class="form-control" name="keterangan[]">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        container.insertAdjacentHTML('beforeend', itemHtml);

        // Tambahkan tombol Tambah Item
        const addButtonHtml = `
            <div class="text-end mt-3">
                <button type="button" class="btn btn-success" onclick="addRincianItem()">
                    <i class="bi bi-plus-circle"></i> Tambah Item
                </button>
            </div>
        `;
        container.insertAdjacentHTML('beforeend', addButtonHtml);
        
        // Initialize money format
        initializeMoneyFormat();
        
        // Tampilkan modal
        const modal = new bootstrap.Modal(document.getElementById('editRincianRiilModal'));
        modal.show();
    }

    function filterRincianBiaya() {
        const searchText = document.getElementById('searchRincianBiaya').value.toLowerCase();
        const cards = document.querySelectorAll('.rincian-biaya-card');
        
        cards.forEach(card => {
            const nama = card.getAttribute('data-nama');
            if (nama.includes(searchText)) {
                card.style.display = '';
            } else {
                card.style.display = 'none';
            }
        });
    }

    function confirmDelete(idSurat) {
        Swal.fire({
            title: 'Hapus Rincian Biaya?',
            text: 'Data yang dihapus tidak dapat dikembalikan!',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Ya, Hapus!',
            cancelButtonText: 'Batal',
            customClass: {
                confirmButton: 'btn btn-danger',
                cancelButton: 'btn btn-secondary'
            },
            buttonsStyling: true,
            reverseButtons: true,
            backdrop: `
                rgba(0,0,0,0.4)
                url("data:image/svg+xml,%3Csvg width='50' height='50' viewBox='0 0 24 24' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath fill='%23fff' d='M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z'/%3E%3C/svg%3E")
                center top
                no-repeat
            `
        }).then((result) => {
            if (result.isConfirmed) {
                // Tampilkan loading state
                Swal.fire({
                    title: 'Menghapus Data...',
                    html: 'Mohon tunggu sebentar',
                    allowOutsideClick: false,
                    showConfirmButton: false,
                    willOpen: () => {
                        Swal.showLoading();
                    }
                });
                
                // Redirect ke halaman hapus
                window.location.href = `hapus_rincian.php?id=${idSurat}`;
            }
        });
    }

    function confirmDeleteRincianRiil(idSurat) {
        Swal.fire({
            title: 'Hapus Rincian Biaya Riil?',
            text: 'Semua data rincian biaya riil untuk surat ini akan dihapus dan tidak dapat dikembalikan!',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Ya, Hapus!',
            cancelButtonText: 'Batal',
            customClass: {
                confirmButton: 'btn btn-danger',
                cancelButton: 'btn btn-secondary'
            },
            buttonsStyling: true,
            reverseButtons: true
        }).then((result) => {
            if (result.isConfirmed) {
                // Tampilkan loading state
                Swal.fire({
                    title: 'Menghapus Data...',
                    html: 'Mohon tunggu sebentar',
                    allowOutsideClick: false,
                    showConfirmButton: false,
                    willOpen: () => {
                        Swal.showLoading();
                    }
                });
                
                // Redirect ke halaman hapus rincian riil
                window.location.href = `hapus_rincian_riil.php?id=${idSurat}`;
            }
        });
    }
    </script>
</body>
</html> 