<?php
session_start();
require_once 'config/config.php';

// Create uploads directory if it doesn't exist
$target_dir = "uploads/";
if (!file_exists($target_dir)) {
    mkdir($target_dir, 0777, true);
    // Create .htaccess to allow access to files
    $htaccess = "Options +Indexes\nOrder allow,deny\nAllow from all";
    file_put_contents($target_dir . ".htaccess", $htaccess);
}

// Get user's bidang if admin_sppd or pimpinan
$user_bidang = '';
if ($_SESSION['level'] === 'admin_sppd' || $_SESSION['level'] === 'pimpinan') {
    $user_id = $_SESSION['user_id'];
    $stmt = mysqli_prepare($conn, "SELECT bidang FROM users WHERE id = ?");
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, "i", $user_id);
        mysqli_stmt_execute($stmt);
        $result_user = mysqli_stmt_get_result($stmt);
        if ($user_data = mysqli_fetch_assoc($result_user)) {
            $user_bidang = $user_data['bidang'];
        }
        mysqli_stmt_close($stmt);
    }
}

// Base query
$base_query = "SELECT st.id_surat, st.no_sppd, st.maksud, st.tanggal_surat, fu.file_name 
               FROM surat_tugas st 
               LEFT JOIN file_uploads fu ON st.id_surat = fu.id_surat";

// Add WHERE clause for admin_sppd and pimpinan
if (($_SESSION['level'] === 'admin_sppd' || $_SESSION['level'] === 'pimpinan') && !empty($user_bidang)) {
    $query = $base_query . " LEFT JOIN users u ON st.created_by = u.id WHERE (u.bidang = ? OR u.bidang IS NULL) ORDER BY st.id_surat DESC";
    $stmt = mysqli_prepare($conn, $query);
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, "s", $user_bidang);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
    } else {
        die("Error preparing statement: " . mysqli_error($conn));
    }
} else {
    $query = $base_query . " ORDER BY st.id_surat DESC";
    $result = mysqli_query($conn, $query);
}

if (!$result) {
    die("Error in query: " . mysqli_error($conn));
}

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Debug information
    error_log("POST data received: " . print_r($_POST, true));
    error_log("FILES data received: " . print_r($_FILES, true));
    
    // Get SPPD number and date
    if (!isset($_POST['id_surat']) || empty($_POST['id_surat'])) {
        $_SESSION['error'] = "ID Surat tidak ditemukan";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }
    
    $id_surat = $_POST['id_surat'];
    error_log("Processing id_surat: " . $id_surat);
    
    // Verify user has access to this surat_tugas
    if ($_SESSION['level'] === 'admin_sppd' || $_SESSION['level'] === 'pimpinan') {
        $check_query = "SELECT st.id_surat FROM surat_tugas st 
                       JOIN users u ON st.created_by = u.id 
                       WHERE st.id_surat = ? AND u.bidang = ?";
        $check_stmt = mysqli_prepare($conn, $check_query);
        mysqli_stmt_bind_param($check_stmt, "is", $id_surat, $user_bidang);
        mysqli_stmt_execute($check_stmt);
        $check_result = mysqli_stmt_get_result($check_stmt);
        
        if (mysqli_num_rows($check_result) === 0) {
            $_SESSION['error'] = "Anda tidak memiliki akses untuk mengunggah file ke SPPD ini.";
            header("Location: " . $_SERVER['PHP_SELF']);
            exit();
        }
    }
    
    // Create uploads directory if it doesn't exist with proper permissions
    if (!file_exists($target_dir)) {
        if (!mkdir($target_dir, 0777, true)) {
            $_SESSION['error'] = "Gagal membuat direktori upload. Silakan hubungi administrator.";
            header("Location: " . $_SERVER['PHP_SELF']);
            exit();
        }
        chmod($target_dir, 0777);
    }

    $get_surat_query = "SELECT no_sppd, tanggal_surat FROM surat_tugas WHERE id_surat = ?";
    $stmt = mysqli_prepare($conn, $get_surat_query);
    if (!$stmt) {
        $_SESSION['error'] = "Error dalam mempersiapkan query: " . mysqli_error($conn);
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }
    
    mysqli_stmt_bind_param($stmt, "i", $id_surat);
    if (!mysqli_stmt_execute($stmt)) {
        $_SESSION['error'] = "Error dalam mengeksekusi query: " . mysqli_error($conn);
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }
    
    $result = mysqli_stmt_get_result($stmt);
    $surat_data = mysqli_fetch_assoc($result);
    
    if (!$surat_data) {
        $_SESSION['error'] = "Data SPPD tidak ditemukan";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }
    
    mysqli_stmt_close($stmt);
    
    // Extract number from SPPD (chars 5-8)
    $sppd_number = substr($surat_data['no_sppd'], 4, 4);
    
    // Get month and year from tanggal_surat
    $bulan = date('m', strtotime($surat_data['tanggal_surat']));
    $tahun = date('Y', strtotime($surat_data['tanggal_surat']));
    
    // Get file extension and validate file type
    $fileType = strtolower(pathinfo($_FILES["fileToUpload"]["name"], PATHINFO_EXTENSION));
    $allowed_types = ['jpg', 'jpeg', 'png', 'gif', 'pdf'];
    
    if (!in_array($fileType, $allowed_types)) {
        $_SESSION['error'] = "Maaf, hanya file JPG, JPEG, PNG, GIF, & PDF yang diperbolehkan.";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }
    
    // Validate file size (50MB max)
    if ($_FILES["fileToUpload"]["size"] > 50 * 1024 * 1024) {
        $_SESSION['error'] = "Maaf, ukuran file terlalu besar (maksimal 50MB).";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }
    
    // Create new filename
    $newFileName = "SPPD_" . $sppd_number . "_" . $bulan . "_" . $tahun . "." . $fileType;
    $target_file = $target_dir . $newFileName;
    
    try {
        // Begin transaction
        mysqli_begin_transaction($conn);
        
        // Check if file already exists and delete it
        if (file_exists($target_file)) {
            if (!unlink($target_file)) {
                throw new Exception("Gagal menghapus file lama.");
            }
        }
        
        // Upload new file
        if (!move_uploaded_file($_FILES["fileToUpload"]["tmp_name"], $target_file)) {
            throw new Exception("Gagal mengupload file.");
        }
        
        // Update database
        $file_path = "uploads/" . $newFileName;
        $update_query = "INSERT INTO file_uploads (id_surat, file_name) VALUES (?, ?) 
                        ON DUPLICATE KEY UPDATE file_name = VALUES(file_name)";
        $stmt = mysqli_prepare($conn, $update_query);
        
        if (!$stmt) {
            throw new Exception("Gagal mempersiapkan query database.");
        }
        
        mysqli_stmt_bind_param($stmt, "is", $id_surat, $file_path);
        
        if (!mysqli_stmt_execute($stmt)) {
            throw new Exception("Gagal menyimpan informasi file ke database.");
        }
        
        // Commit transaction
        mysqli_commit($conn);
        
        $_SESSION['success'] = "File berhasil diunggah dengan nama: " . $newFileName;
        
    } catch (Exception $e) {
        // Rollback transaction on error
        mysqli_rollback($conn);
        $_SESSION['error'] = "Error: " . $e->getMessage();
    }
    
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload Berkas - SPPD System</title>
    <?php include 'includes/header.php' ?>
    
    <!-- Tambahkan CSS untuk styling -->
    <style>
        body {
            background: linear-gradient(135deg,rgb(245, 255, 254) 0%,rgb(153, 224, 238) 100%);
        }

        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 0 20px rgba(0,0,0,0.05);
        }
        
        .card-header {
            background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
            border-radius: 15px 15px 0 0 !important;
            padding: 20px;
            border: none;
        }
        
        .card-header h5 {
            color: white;
            font-weight: 600;
            margin: 0;
            font-size: 1.25rem;
        }

        .table {
            margin-bottom: 0;
            width: 100%;
            background-color: #fff;
            border-radius: 8px;
            overflow: hidden;
        }

        .table thead th {
            background: linear-gradient(135deg, #0ea5e9 0%, #2dd4bf 100%);
            color: #fff;
            font-weight: 500;
            padding: 15px;
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border: none;
            vertical-align: middle;
        }

        .table tbody tr {
            transition: all 0.3s ease;
        }

        .table tbody tr:hover {
            background-color: rgba(13, 110, 253, 0.05);
            transform: scale(1.01);
        }

        .table td {
            padding: 12px 15px;
            vertical-align: middle;
            border-bottom: 1px solid #e9ecef;
            font-size: 0.9rem;
            color: #495057;
        }

        .btn-upload {
            color: #198754;
            background-color: rgba(25, 135, 84, 0.1);
            border: 1px solid rgba(25, 135, 84, 0.2);
            transition: all 0.3s ease;
            padding: 8px 16px;
            border-radius: 8px;
            font-weight: 500;
        }

        .btn-upload:hover {
            color: #fff;
            background-color: #198754;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(25, 135, 84, 0.2);
        }

        .btn-view {
            color: #0d6efd;
            background-color: rgba(13, 110, 253, 0.1);
            border: 1px solid rgba(13, 110, 253, 0.2);
            transition: all 0.3s ease;
            padding: 8px 16px;
            border-radius: 8px;
            font-weight: 500;
        }

        .btn-view:hover {
            color: #fff;
            background-color: #0d6efd;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(13, 110, 253, 0.2);
        }

        .home-section {
            background: linear-gradient(135deg,rgb(231, 255, 255) 0%,rgb(236, 255, 246) 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .btn i {
            margin-right: 5px;
            font-size: 0.875rem;
            vertical-align: middle;
        }
    </style>
</head>
<body>
    <?php include 'includes/sidebar.php' ?>
    
    <section class="home-section">
        <div class="container mt-4">
            <div class="card mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Upload Berkas SPPD</h5>
                </div>
                <div class="card-body">
                    <?php if (isset($_SESSION['success'])): ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <?= $_SESSION['success'] ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                        </div>
                        <?php unset($_SESSION['success']); ?>
                    <?php endif; ?>

                    <?php if (isset($_SESSION['error'])): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <?= $_SESSION['error'] ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                        </div>
                        <?php unset($_SESSION['error']); ?>
                    <?php endif; ?>

                    <div class="table-responsive">
                        <table id="sppdTable" class="table table-hover">
                            <thead>
                                <tr>
                                    <th class="text-center" style="width: 5%">NO.</th>
                                    <th style="width: 25%">NO. SPPD</th>
                                    <th style="width: 50%">MAKSUD</th>
                                    <th style="width: 20%" class="text-center">AKSI</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $no = 1;
                                while ($row = mysqli_fetch_assoc($result)): 
                                ?>
                                <tr>
                                    <td class="text-center"><?= $no++ ?></td>
                                    <td><?= htmlspecialchars($row['no_sppd']) ?></td>
                                    <td><?= htmlspecialchars($row['maksud']) ?></td>
                                    <td class="text-center">
                                        <?php if (empty($row['file_name'])): ?>
                                            <button type="button" class="btn btn-upload btn-sm" data-bs-toggle="modal" data-bs-target="#uploadModal" data-id="<?= $row['id_surat'] ?>" data-sppd="<?= htmlspecialchars($row['no_sppd']) ?>">
                                                <i class="bi bi-cloud-upload"></i> Upload
                                            </button>
                                        <?php else: ?>
                                            <a href="detail_upload.php?id=<?= $row['id_surat'] ?>" class="btn btn-view btn-sm">
                                                <i class="bi bi-eye"></i> Lihat
                                            </a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Upload Modal -->
    <div class="modal fade" id="uploadModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Upload Berkas SPPD</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form action="" method="post" enctype="multipart/form-data" id="uploadForm">
                    <div class="modal-body">
                        <input type="hidden" name="id_surat" id="id_surat">
                        <div class="mb-3">
                            <label class="form-label">Nomor SPPD</label>
                            <input type="text" class="form-control" id="no_sppd" readonly>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Pilih Berkas</label>
                            <input type="file" class="form-control" name="fileToUpload" required accept=".jpg,.jpeg,.png,.gif,.pdf">
                            <div class="form-text">Format: JPG, JPEG, PNG, GIF, PDF (Maks. 50MB)</div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-primary">Upload</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <?php include 'includes/footer.php' ?>

    <script>
        $(document).ready(function() {
            $('#sppdTable').DataTable({
                "responsive": true,
                "language": {
                    "url": "//cdn.datatables.net/plug-ins/1.10.24/i18n/Indonesian.json"
                },
                "pageLength": 10,
                "lengthMenu": [[10, 25, 50, -1], [10, 25, 50, "Semua"]]
            });

            // Handle upload button click
            $(document).on('click', '.btn-upload', function(e) {
                e.preventDefault();
                var id = $(this).data('id');
                var sppd = $(this).data('sppd');
                
                // Set the values
                $('#id_surat').val(id);
                $('#no_sppd').val(sppd);
                
                // For debugging
                console.log('ID Surat:', id);
                console.log('No SPPD:', sppd);
            });

            // Form submission handler
            $('#uploadForm').on('submit', function(e) {
                e.preventDefault();
                
                var id_surat = $('#id_surat').val();
                var fileInput = $('input[name="fileToUpload"]')[0];
                
                if (!id_surat) {
                    alert('Error: ID Surat tidak ditemukan');
                    return false;
                }
                
                if (!fileInput.files.length) {
                    alert('Silakan pilih file terlebih dahulu');
                    return false;
                }
                
                // Create FormData object
                var formData = new FormData(this);
                
                // Submit form using AJAX
                $.ajax({
                    url: window.location.href,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        // Reload the page after successful upload
                        window.location.reload();
                    },
                    error: function(xhr, status, error) {
                        alert('Terjadi kesalahan saat upload file: ' + error);
                    }
                });
            });
        });
    </script>
</body>
</html> 